"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * adgm service
 */
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const bcrypt = require("bcryptjs");
const { v4: uuidv4 } = require("uuid");
async function generateOrderNum(type) {
    try {
        const uuid = uuidv4();
        const base36 = parseInt(uuid.replace(/-/g, ""), 16).toString(36);
        let firstLetter = type === "others" ? "O" : "A";
        const orderNumber = firstLetter + base36.slice(0, 7).toUpperCase();
        const isOrderNumberExists = await strapi.db
            .query("api::order.order")
            .findOne({
            where: { orderId: orderNumber, is_deleted: false },
        });
        if (isOrderNumberExists)
            return await generateOrderNum(type);
        return orderNumber;
    }
    catch (error) {
        console.log("error in generating order number", error);
    }
}
exports.default = strapi_1.factories.createCoreService("api::adgm.adgm", ({ strapi }) => ({
    async findExportData() {
        try {
            const data = await strapi.db.query("api::adgm.adgm").findMany({
                where: { is_deleted: false },
                orderBy: { id: "desc" },
                populate: true,
            });
            return new dataResponse_dto_1.DataResponseDto(data, "Adgm data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllAdgms(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                is_deleted: false,
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search &&
                    search.trim().length && {
                    $or: [
                        { name: { $containsi: search } },
                        { adgm_id: { $containsi: search } },
                    ],
                }),
            };
            const data = await strapi.db.query("api::adgm.adgm").findMany({
                where,
                offset,
                limit: limit,
                orderBy: { id: "desc" },
                populate: {
                    orderId: {
                        populate: {
                            creator: {
                                select: ["username", "firstName", "lastName"]
                            }
                        }
                    }
                },
            });
            if (data) {
                const totalDataCount = await strapi
                    .query("api::adgm.adgm")
                    .count({ where });
                const pageCount = Math.ceil(totalDataCount / limit);
                return new dataResponse_dto_1.DataResponseDto({
                    data,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount,
                        total: totalDataCount,
                    },
                }, "Adgm data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    getByUUID: async (id) => {
        var _a;
        try {
            const adgm = await strapi.query("api::adgm.adgm").findOne({
                where: { adgm_id: id },
                populate: {
                    orderId: true,
                    ticket_masters: {
                        where: {
                            is_deleted: false,
                            status: {
                                $notIn: ["deleted", "refunded", "cancelled"]
                            }
                        },
                        populate: {
                            ticket_id: true,
                        },
                        select: [
                            "id",
                            "firstName",
                            "lastName",
                            "email",
                            "day",
                            "ticketNumber",
                            "token",
                            "status",
                        ],
                    },
                },
            });
            return new dataResponse_dto_1.DataResponseDto(adgm, "adgm fetched successfully");
        }
        catch (error) {
            console.log(error);
            if ((_a = error.details) === null || _a === void 0 ? void 0 : _a.errors) {
                return new dataResponse_dto_1.DataResponseDto(null, false, error.details.errors);
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
            }
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db.query("api::adgm.adgm").update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "adgm deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async adgmLogin(body) {
        try {
            const { adgmId, password } = body;
            if (!adgmId || !password) {
                return new dataResponse_dto_1.DataResponseDto(null, "Please provide valid credentials");
            }
            const adgm = await strapi.query("api::adgm.adgm").findOne({
                where: { adgm_id: adgmId, is_deleted: false },
            });
            if (!adgm) {
                return {
                    data: null,
                    status: false,
                    message: "Adgm not found.",
                };
            }
            const isPasswordValid = await bcrypt.compare(password, adgm === null || adgm === void 0 ? void 0 : adgm.password);
            if (!isPasswordValid) {
                return {
                    data: null,
                    status: false,
                    message: "Invalid credentials.",
                };
            }
            return new dataResponse_dto_1.DataResponseDto(adgm, "Sponsor logged in successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    createAdgm: async (body) => {
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                var _a;
                if (!body.password) {
                    return new dataResponse_dto_1.DataResponseDto(null, "Password is required");
                }
                let generatedOrderNumber = await generateOrderNum("others");
                const order = await strapi.entityService.create("api::order.order", {
                    data: {
                        orderId: generatedOrderNumber,
                        type: body === null || body === void 0 ? void 0 : body.type,
                        paymentId: null,
                        eventDate: new Date(),
                        firstName: body === null || body === void 0 ? void 0 : body.username,
                        lastName: (body === null || body === void 0 ? void 0 : body.lastName) || "",
                        email: body === null || body === void 0 ? void 0 : body.email,
                        company: (body === null || body === void 0 ? void 0 : body.company) || "",
                        sector: (body === null || body === void 0 ? void 0 : body.sector) || "",
                        designation: (body === null || body === void 0 ? void 0 : body.designation) || "",
                        nationality: (body === null || body === void 0 ? void 0 : body.nationality) || "",
                        residanceCountry: (body === null || body === void 0 ? void 0 : body.residanceCountry) || "",
                        countryCode: (body === null || body === void 0 ? void 0 : body.countryCode) || "",
                        mobile: (body === null || body === void 0 ? void 0 : body.mobile) || "",
                        photo: (body === null || body === void 0 ? void 0 : body.photo) || "",
                        ticketCount: (body === null || body === void 0 ? void 0 : body.ticketCount) || 0,
                        discountAmount: 0,
                        totalAmount: 0,
                        paidAmount: 0,
                        is_discounted: false,
                        publishedAt: new Date(),
                        utm_source: (body === null || body === void 0 ? void 0 : body.utm_source) || "",
                        utm_medium: (body === null || body === void 0 ? void 0 : body.utm_medium) || "",
                        utm_campaign: (body === null || body === void 0 ? void 0 : body.utm_campaign) || "",
                        utm_content: (body === null || body === void 0 ? void 0 : body.utm_content) || "",
                        utm_id: (body === null || body === void 0 ? void 0 : body.utm_id) || "",
                        userType: "adgm",
                        status: "Free",
                        creator: body === null || body === void 0 ? void 0 : body.userId,
                        modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                    },
                    transacting: trx,
                });
                let hashPassword = await bcrypt.hash((_a = body === null || body === void 0 ? void 0 : body.password) === null || _a === void 0 ? void 0 : _a.toString().trim(), 10);
                let generatedAdgmNumber = await generateOrderNum("adgm");
                let adgm = {
                    name: body === null || body === void 0 ? void 0 : body.name,
                    orderId: order === null || order === void 0 ? void 0 : order.id,
                    adgm_id: generatedAdgmNumber,
                    executive: (body === null || body === void 0 ? void 0 : body.executive) || 0,
                    delegate: (body === null || body === void 0 ? void 0 : body.delegate) || 0,
                    paddock_club: (body === null || body === void 0 ? void 0 : body.paddock_club) || 0,
                    general_admission: (body === null || body === void 0 ? void 0 : body.general_admission) || 0,
                    resolve: (body === null || body === void 0 ? void 0 : body.resolve) || 0,
                    asset_abu_dhabi: (body === null || body === void 0 ? void 0 : body.asset_abu_dhabi) || 0,
                    fintech_abu_dhabi: (body === null || body === void 0 ? void 0 : body.fintech_abu_dhabi) || 0,
                    adsff: (body === null || body === void 0 ? void 0 : body.adsff) || 0,
                    single_day: (body === null || body === void 0 ? void 0 : body.single_day) || 0,
                    remaining_single_day_delegate_count: (body === null || body === void 0 ? void 0 : body.single_day_delegate) || 0,
                    remaining_executive: (body === null || body === void 0 ? void 0 : body.executive) || 0,
                    remaining_delegate: (body === null || body === void 0 ? void 0 : body.delegate) || 0,
                    remaining_paddock: (body === null || body === void 0 ? void 0 : body.paddock_club) || 0,
                    remainig_general_admission: (body === null || body === void 0 ? void 0 : body.general_admission) || 0,
                    remaining_resolve: (body === null || body === void 0 ? void 0 : body.resolve) || 0,
                    remaining_asset_abu_dhabi: (body === null || body === void 0 ? void 0 : body.asset_abu_dhabi) || 0,
                    remaining_fintech_abu_dhabi: (body === null || body === void 0 ? void 0 : body.fintech_abu_dhabi) || 0,
                    remaining_adsff: (body === null || body === void 0 ? void 0 : body.adsff) || 0,
                    remaining_single_day: (body === null || body === void 0 ? void 0 : body.single_day) || 0,
                    is_deleted: false,
                    password: hashPassword,
                    publishedAt: new Date(),
                    creator: body === null || body === void 0 ? void 0 : body.userId,
                };
                const adgmData = await strapi.entityService.create("api::adgm.adgm", {
                    data: adgm,
                    transacting: trx,
                });
                let res = {
                    order,
                    adgmData,
                };
                return new dataResponse_dto_1.DataResponseDto(res, "Order Created successfully");
            });
            return result;
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    updateAdgm: async (id, body) => {
        try {
            const existingAdgm = await strapi.db.query("api::adgm.adgm").findOne({
                where: { id },
            });
            if (!existingAdgm) {
                return new dataResponse_dto_1.DataResponseDto(null, "Adgm not found");
            }
            const order = await strapi.db.query("api::order.order").update({
                where: { orderId: body.orderId },
                data: {
                    firstName: body === null || body === void 0 ? void 0 : body.usename,
                    lastName: (body === null || body === void 0 ? void 0 : body.lastName) || "",
                    email: (body === null || body === void 0 ? void 0 : body.email) || "",
                    company: (body === null || body === void 0 ? void 0 : body.company) || "",
                    modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                },
            });
            let remaining_executive_count = Number(existingAdgm.remaining_executive);
            let remaining_delegate_count = Number(existingAdgm.remaining_delegate);
            let remaining_padock_club_count = Number(existingAdgm.remaining_paddock);
            let remaining_general_admission_count = Number(existingAdgm.remainig_general_admission);
            let remaining_single_day_delegate_count = Number(existingAdgm.remaining_single_day);
            let remaining_resolve_count = Number(existingAdgm.remaining_resolve);
            let remaining_asset_abu_dhabi_count = Number(existingAdgm.remaining_asset_abu_dhabi);
            let remaining_fintech_abu_dhabi_count = Number(existingAdgm.remaining_fintech_abu_dhabi);
            let remaining_adsff_count = Number(existingAdgm.remaining_adsff);
            // exective
            if (body.executive === 0 || body.executive) {
                const newExecutiveCount = body.executive === 0 ? 0 : body.executive;
                const executive_remaining = Number(existingAdgm.executive) -
                    Number(existingAdgm.remaining_executive);
                if (Number(executive_remaining) > Number(body.executive)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "executive cannot be less than redeemed tickets count");
                }
                remaining_executive_count =
                    Number(newExecutiveCount) - Number(executive_remaining);
            }
            // full week
            if (body.delegate === 0 || body.delegate) {
                const newDelegateCount = body.delegate === 0 ? 0 : body.delegate;
                let delegate_remaining = Number(existingAdgm.delegate) -
                    Number(existingAdgm.remaining_delegate);
                if (Number(delegate_remaining) > Number(body.delegate)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total full week count cannot be less than redeemed tickets count");
                }
                remaining_delegate_count =
                    Number(newDelegateCount) - Number(delegate_remaining);
            }
            // padock_club
            if (body.padock_club === 0 || body.padock_club) {
                const newPadockCount = body.padock_club === 0 ? 0 : body.padock_club;
                let padock_club_remaining = Number(existingAdgm.padock_club) -
                    Number(existingAdgm.remaining_paddock);
                if (Number(padock_club_remaining) > Number(body.padock_club)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total padock_club count cannot be less than redeemed tickets count");
                }
                remaining_padock_club_count =
                    Number(newPadockCount) - Number(padock_club_remaining);
            }
            // general_admission
            if (body.general_admission === 0 || body.general_admission) {
                const newGeneralCount = body.general_admission === 0 ? 0 : body.general_admission;
                let general_admission_remaining = Number(existingAdgm.general_admission) -
                    Number(existingAdgm.remainig_general_admission);
                if (Number(general_admission_remaining) > Number(body.general_admission)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total general_admission count cannot be less than redeemed tickets count");
                }
                remaining_general_admission_count =
                    Number(newGeneralCount) -
                        Number(general_admission_remaining);
            }
            // single_day_delegate
            if (body.single_day === 0 || body.single_day) {
                const newSingleDayCount = body.single_day === 0 ? 0 : body.single_day;
                let single_day_delegate_remaining = Number(existingAdgm.single_day) -
                    Number(existingAdgm.remaining_single_day);
                if (Number(single_day_delegate_remaining) >
                    Number(body.single_day)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total single_day_delegate count cannot be less than redeemed tickets count");
                }
                remaining_single_day_delegate_count =
                    Number(newSingleDayCount) - Number(single_day_delegate_remaining);
            }
            // resolve done
            if (body.resolve === 0 || body.resolve) {
                const newResolveCount = body.resolve === 0 ? 0 : body.resolve;
                let resolve_remaining = Number(existingAdgm.resolve) -
                    Number(existingAdgm.remaining_resolve);
                if (Number(resolve_remaining) > Number(body.resolve)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total resolve count cannot be less than redeemed tickets count");
                }
                remaining_resolve_count =
                    Number(newResolveCount) - Number(resolve_remaining);
            }
            // asset_abu_dhabi
            if (body.asset_abu_dhabi === 0 || body.asset_abu_dhabi) {
                const newAssetAbhuDhabi = body.asset_abu_dhabi === 0 ? 0 : body.asset_abu_dhabi;
                let asset_abu_dhabi_remaining = Number(existingAdgm.asset_abu_dhabi) -
                    Number(existingAdgm.remaining_asset_abu_dhabi);
                if (Number(asset_abu_dhabi_remaining) > Number(body.asset_abu_dhabi)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total asset_abu_dhabi count cannot be less than redeemed tickets count");
                }
                remaining_asset_abu_dhabi_count =
                    Number(newAssetAbhuDhabi) - Number(asset_abu_dhabi_remaining);
            }
            // fintech_abu_dhabi
            if (body.fintech_abu_dhabi === 0 || body.fintech_abu_dhabi) {
                const newFintechCount = body.fintech_abu_dhabi === 0 ? 0 : body.fintech_abu_dhabi;
                let fintech_abu_dhabi_remaining = Number(existingAdgm.fintech_abu_dhabi) -
                    Number(existingAdgm.remaining_fintech_abu_dhabi);
                if (Number(fintech_abu_dhabi_remaining) > Number(body.fintech_abu_dhabi)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total fintech_abu_dhabi count cannot be less than redeemed tickets count");
                }
                remaining_fintech_abu_dhabi_count =
                    Number(newFintechCount) -
                        Number(fintech_abu_dhabi_remaining);
            }
            // adsff
            if (body.adsff === 0 || body.adsff) {
                const newAdsffCount = body.adsff === 0 ? 0 : body.adsff;
                let adsff_remaining = Number(existingAdgm.adsff) - Number(existingAdgm.remaining_adsff);
                if (Number(adsff_remaining) > Number(body.adsff)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "total adsff count cannot be less than redeemed tickets count");
                }
                remaining_adsff_count =
                    Number(newAdsffCount) - Number(adsff_remaining);
            }
            let updateDataObj = {
                name: body === null || body === void 0 ? void 0 : body.name,
                password: existingAdgm.password,
                paddock_club: Number(existingAdgm.paddock_club),
                executive: Number(body === null || body === void 0 ? void 0 : body.executive),
                delegate: Number(body === null || body === void 0 ? void 0 : body.delegate),
                general_admission: Number(body === null || body === void 0 ? void 0 : body.general_admission),
                resolve: Number(body === null || body === void 0 ? void 0 : body.resolve),
                asset_abu_dhabi: Number(body === null || body === void 0 ? void 0 : body.asset_abu_dhabi),
                fintech_abu_dhabi: Number(body === null || body === void 0 ? void 0 : body.fintech_abu_dhabi),
                adsff: Number(body === null || body === void 0 ? void 0 : body.adsff),
                single_day: Number(body === null || body === void 0 ? void 0 : body.single_day),
                remaining_executive: Number(remaining_executive_count),
                remaining_delegate: Number(remaining_delegate_count),
                remaining_paddock: Number(remaining_padock_club_count),
                remainig_general_admission: Number(remaining_general_admission_count),
                remaining_resolve: Number(remaining_resolve_count),
                remaining_asset_abu_dhabi: Number(remaining_asset_abu_dhabi_count),
                remaining_fintech_abu_dhabi: Number(remaining_fintech_abu_dhabi_count),
                remaining_adsff: Number(remaining_adsff_count),
                remaining_single_day: Number(remaining_single_day_delegate_count),
                modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
            };
            if ((body === null || body === void 0 ? void 0 : body.paddock_club) === 0 || (body === null || body === void 0 ? void 0 : body.paddock_club)) {
                updateDataObj.paddock_club = (body === null || body === void 0 ? void 0 : body.paddock_club) === 0 ? 0 : Number(body === null || body === void 0 ? void 0 : body.paddock_club);
            }
            if (body === null || body === void 0 ? void 0 : body.password) {
                updateDataObj.password = await bcrypt.hash(body.password.toString().trim(), 10);
            }
            ;
            // update adgm
            const updatedAdgm = await strapi.db.query("api::adgm.adgm").update({
                where: { id },
                data: updateDataObj
            });
            let res = {
                order,
                updatedAdgm,
            };
            return new dataResponse_dto_1.DataResponseDto(res, "adgm Updated successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
}));
