"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const moment_1 = __importDefault(require("moment"));
const uuid_1 = require("uuid");
exports.default = strapi_1.factories.createCoreService("api::agenda-session.agenda-session", ({ strapi }) => ({
    async findExportData() {
        try {
            const data = await strapi.db.query("api::agenda-session.agenda-session").findMany({
                where: {
                    is_deleted: false
                },
                populate: true,
                orderBy: { date: "asc" },
            });
            return new dataResponse_dto_1.DataResponseDto(data, "Agenda sessions data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllSessions(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                is_deleted: false,
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search && search.trim().length && {
                    $or: [
                        { title: { $containsi: search } },
                        { description: { $containsi: search } },
                    ],
                }),
            };
            const data = await strapi.db.query("api::agenda-session.agenda-session").findMany({
                where,
                populate: {
                    creator: true,
                    modifiedBy: true
                },
                offset,
                limit: limit,
                orderBy: { id: "desc" },
            });
            if (data) {
                const totalDataCount = await strapi
                    .query("api::agenda-session.agenda-session")
                    .count({ where });
                const pageCount = Math.ceil(totalDataCount / limit);
                return new dataResponse_dto_1.DataResponseDto({
                    data,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount,
                        total: totalDataCount,
                    },
                }, "agenda-session data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findOneSession(id, params) {
        var _a;
        try {
            const { ticketId } = params || {};
            const data = await strapi.db.query("api::agenda-session.agenda-session").findOne({
                where: {
                    id,
                    is_deleted: false
                },
                select: ["id", "title", "description", "fromTime", "toTime", "day", "date", "color", "image", "video", "published", "publishVideo", "isBreak"],
                populate: {
                    agenda: {
                        populate: {
                            location: {
                                select: ["name"]
                            },
                        },
                        select: ["id", "title", "date", "logo", "mobileBanner", "agendaMobileBanner", "color"]
                    },
                    sessionType: {
                        select: ["name", "icon"]
                    },
                    moderator: {
                        where: {
                            is_deleted: false,
                        },
                        select: ["firstName", "lastName", "photoUrl", "companyName", "designation", "bio"]
                    },
                    location: {
                        select: ["name"]
                    },
                    speakers: {
                        where: {
                            is_deleted: false,
                            published: true
                        },
                        select: ["firstName", "lastName", "photoUrl", "companyName", "designation", "bio"]
                    },
                    favourites: {
                        where: {
                            ...(ticketId && { ticket_id: ticketId }),
                            is_deleted: false
                        }
                    }
                }
            });
            const { favourites, ...updatedData } = data;
            if (ticketId) {
                updatedData.isFavourite = ((_a = data === null || data === void 0 ? void 0 : data.favourites) === null || _a === void 0 ? void 0 : _a.length) ? true : false;
            }
            return new dataResponse_dto_1.DataResponseDto(updatedData, "Session data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async datewiseSessionsSpeakers() {
        try {
            const agendaSessions = await strapi
                .query("api::agenda-session.agenda-session")
                .findMany({
                where: {
                    is_deleted: false,
                },
                populate: {
                    speakers: true,
                },
            });
            const groupedSpeakersByDate = agendaSessions.reduce((acc, session) => {
                let sessionDate = new Date(session.date);
                const date = sessionDate.toISOString().split("T")[0];
                let dateGroup = acc.find((group) => group.date === date);
                if (!dateGroup) {
                    dateGroup = { date, session, speakers: [] };
                    acc.push(dateGroup);
                }
                const speakersToAdd = session.speakers.slice(0, 5 - dateGroup.speakers.length);
                dateGroup['sessionData'] = {
                    day: session.day,
                    date: session.date,
                    title: session.title,
                    description: session.description,
                    image: session.image,
                    isBreak: session.isBreak,
                    published: session.published,
                    fromTime: session.fromTime,
                    toTime: session.toTime,
                    color: session.color
                };
                dateGroup.speakers.push(...speakersToAdd);
                return acc;
            }, []);
            return new dataResponse_dto_1.DataResponseDto(groupedSpeakersByDate, "Speakers data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db
                .query("api::agenda-session.agenda-session")
                .update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                    agenda: []
                },
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "Agenda session deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async generateSessionICSFile(id) {
        var _a, _b;
        try {
            const sessionData = await strapi.db.query("api::agenda-session.agenda-session").findOne({
                where: {
                    id,
                    is_deleted: false
                },
                populate: {
                    agenda: {
                        select: ["date"],
                        populate: {
                            location: {
                                select: ["name"]
                            }
                        }
                    }
                }
            });
            if (!sessionData) {
                return new dataResponse_dto_1.DataResponseDto(null, "No session found");
            }
            // Format dates
            const format_from = 'YYYY-MM-DD h:mm A';
            const format_to = 'YYYYMMDDTHHmmss';
            const date = sessionData.agenda.date;
            const startTime = sessionData === null || sessionData === void 0 ? void 0 : sessionData.fromTime;
            const endTime = sessionData === null || sessionData === void 0 ? void 0 : sessionData.toTime;
            const DTSTAMP = (0, moment_1.default)().format(format_to) + 'Z';
            const DTSTART = (0, moment_1.default)(`${date} ${startTime}`, format_from).format(format_to);
            const DTEND = (0, moment_1.default)(`${date} ${endTime}`, format_from).format(format_to);
            const UID = `${(0, uuid_1.v4)()}@adfw.com`;
            const icsContent = [
                'BEGIN:VCALENDAR',
                'VERSION:2.0',
                'PRODID:-//ADFW//Meeting Calendar//EN',
                'CALSCALE:GREGORIAN',
                'BEGIN:VTIMEZONE',
                'TZID:Asia/Dubai',
                'TZURL:http://tzurl.org/zoneinfo-outlook/Asia/Dubai',
                'X-LIC-LOCATION:Asia/Dubai',
                'BEGIN:STANDARD',
                'TZOFFSETFROM:+0400',
                'TZOFFSETTO:+0400',
                'TZNAME:+04',
                'DTSTART:19700101T000000',
                'RRULE:FREQ=YEARLY;BYMONTH=10;BYDAY=-1SU',
                'END:STANDARD',
                'END:VTIMEZONE',
                'BEGIN:VEVENT',
                `DTSTAMP:${DTSTAMP}`,
                `UID:${UID}`,
                `DTSTART;TZID=Asia/Dubai:${DTSTART}`,
                `DTEND;TZID=Asia/Dubai:${DTEND}`,
                `SUMMARY:${sessionData.title}`,
                `DESCRIPTION:  ${sessionData === null || sessionData === void 0 ? void 0 : sessionData.description}`,
                `LOCATION:${((_b = (_a = sessionData === null || sessionData === void 0 ? void 0 : sessionData.agenda) === null || _a === void 0 ? void 0 : _a.location) === null || _b === void 0 ? void 0 : _b.name) || 'TBD'}`,
                'BEGIN:VALARM',
                'ACTION:DISPLAY',
                `DESCRIPTION:Reminder: ${sessionData === null || sessionData === void 0 ? void 0 : sessionData.description}`,
                'TRIGGER:-PT5M',
                'END:VALARM',
                'END:VEVENT',
                'END:VCALENDAR'
            ].join('\r\n');
            return icsContent.trim();
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    // public session api
    async findSessions(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                published: true,
                isBreak: false,
                is_deleted: false,
                agenda: {
                    $notNull: true,
                },
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search && search.trim().length && {
                    $or: [
                        { title: { $containsi: search } },
                        { description: { $containsi: search } },
                    ],
                }),
            };
            const data = await strapi.db.query("api::agenda-session.agenda-session").findMany({
                where,
                // offset,
                // limit: limit,
                orderBy: [
                    { agenda: { date: "asc" } },
                    { toTime: "asc" }
                ],
                select: ["title", "fromTime", "toTime", "day", "published"],
                populate: {
                    agenda: {
                        populate: {
                            location: {
                                select: ["name"]
                            }
                        },
                        select: ["title", "date"]
                    },
                    sessionType: {
                        select: ["name"]
                    },
                    moderator: {
                        select: ["firstName", "lastName", "photoUrl"]
                    },
                    speakers: {
                        select: ["firstName", "lastName", "photoUrl"]
                    },
                }
            });
            const formattedData = data.map((item) => {
                var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
                return {
                    session_name: item === null || item === void 0 ? void 0 : item.title,
                    session_type: (_a = item === null || item === void 0 ? void 0 : item.sessionType) === null || _a === void 0 ? void 0 : _a.name,
                    session_id: item.id,
                    start_time: item === null || item === void 0 ? void 0 : item.fromTime,
                    end_time: item === null || item === void 0 ? void 0 : item.toTime,
                    speakers: item.speakers.map((item) => {
                        return {
                            full_name: (item === null || item === void 0 ? void 0 : item.firstName) || (item === null || item === void 0 ? void 0 : item.lastName) ? `${(item === null || item === void 0 ? void 0 : item.firstName) || ""} ${(item === null || item === void 0 ? void 0 : item.lastName) || ""}`.trim() : "",
                            profile_url: item === null || item === void 0 ? void 0 : item.photoUrl
                        };
                    }),
                    moderator: {
                        full_name: ((_b = item === null || item === void 0 ? void 0 : item.moderator) === null || _b === void 0 ? void 0 : _b.firstName) || ((_c = item === null || item === void 0 ? void 0 : item.moderator) === null || _c === void 0 ? void 0 : _c.lastName)
                            ? `${((_d = item === null || item === void 0 ? void 0 : item.moderator) === null || _d === void 0 ? void 0 : _d.firstName) || ""} ${((_e = item === null || item === void 0 ? void 0 : item.moderator) === null || _e === void 0 ? void 0 : _e.lastName) || ""}`.trim()
                            : "",
                        profile_url: (_f = item === null || item === void 0 ? void 0 : item.moderator) === null || _f === void 0 ? void 0 : _f.photoUrl
                    },
                    event: (_g = item === null || item === void 0 ? void 0 : item.agenda) === null || _g === void 0 ? void 0 : _g.title,
                    day: new Date((_h = item === null || item === void 0 ? void 0 : item.agenda) === null || _h === void 0 ? void 0 : _h.date).toISOString().split('T')[0],
                    location: (_k = (_j = item === null || item === void 0 ? void 0 : item.agenda) === null || _j === void 0 ? void 0 : _j.location) === null || _k === void 0 ? void 0 : _k.name
                };
            });
            return {
                data: formattedData,
                status: true,
                message: "Sessions data fetched successfully"
            };
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
}));
