"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const moment_1 = __importDefault(require("moment"));
exports.default = strapi_1.factories.createCoreService("api::agenda.agenda", ({ strapi }) => ({
    async findExportData() {
        try {
            const data = await strapi.db.query("api::agenda.agenda").findMany({
                where: { is_deleted: false },
                orderBy: { date: "asc" },
                populate: true,
            });
            return new dataResponse_dto_1.DataResponseDto(data, "Agenda data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllAgendas(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                is_deleted: false,
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search &&
                    search.trim().length && {
                    $or: [{ title: { $containsi: search } }],
                }),
            };
            const data = await strapi.db.query("api::agenda.agenda").findMany({
                where,
                populate: {
                    creator: true,
                    modifiedBy: true
                },
                offset,
                limit: limit,
                orderBy: { id: "desc" },
            });
            if (data) {
                const totalDataCount = await strapi
                    .query("api::agenda.agenda")
                    .count({ where });
                const pageCount = Math.ceil(totalDataCount / limit);
                return new dataResponse_dto_1.DataResponseDto({
                    data,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount,
                        total: totalDataCount,
                    },
                }, "Agenda data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async picker(query) {
        try {
            const data = await strapi.db.query("api::agenda.agenda").findMany({
                where: query
                    ? {
                        is_deleted: false,
                        title: { $contains: query },
                    }
                    : { is_deleted: false },
                select: ["id", "title"],
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "Agenda data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async getAgendaDetails(id) {
        var _a, _b, _c, _d, _e;
        try {
            const agenda = await strapi.query("api::agenda.agenda").findOne({
                where: { id, is_deleted: false },
                populate: {
                    sponsers: true,
                    agendaType: true,
                    location: true,
                    agenda_sessions: {
                        populate: {
                            speakers: true,
                            location: true,
                            sessionType: true,
                            moderator: true,
                        },
                    },
                },
            });
            if (!agenda)
                return new dataResponse_dto_1.DataResponseDto(null, "Agenda not found");
            // sorted sessions
            const sessionArray = (_a = agenda === null || agenda === void 0 ? void 0 : agenda.agenda_sessions) === null || _a === void 0 ? void 0 : _a.sort((a, b) => {
                const dateA = a.date ? new Date(a.date) : new Date(0); // Use epoch if date is null
                const dateB = b.date ? new Date(b.date) : new Date(0);
                // Compare by date first
                if (dateA.getTime() !== dateB.getTime()) {
                    return dateA.getTime() - dateB.getTime();
                }
                // If dates are the same, compare by fromTime
                function parseTime(timeStr) {
                    const [hours, minutes] = timeStr.split(':').map(Number);
                    const date = new Date();
                    date.setHours(hours, minutes, 0, 0);
                    return date;
                }
                const timeA = parseTime(a.fromTime);
                const timeB = parseTime(b.fromTime);
                return timeA.getTime() - timeB.getTime();
            }).map((session) => {
                var _a, _b, _c, _d;
                return {
                    id: session === null || session === void 0 ? void 0 : session.id,
                    video: session === null || session === void 0 ? void 0 : session.video,
                    day: session === null || session === void 0 ? void 0 : session.day,
                    date: session === null || session === void 0 ? void 0 : session.date,
                    image: session === null || session === void 0 ? void 0 : session.image,
                    title: session === null || session === void 0 ? void 0 : session.title,
                    description: session === null || session === void 0 ? void 0 : session.description,
                    isBreak: session === null || session === void 0 ? void 0 : session.isBreak,
                    published: session === null || session === void 0 ? void 0 : session.published,
                    fromTime: session === null || session === void 0 ? void 0 : session.fromTime,
                    toTime: session === null || session === void 0 ? void 0 : session.toTime,
                    color: session === null || session === void 0 ? void 0 : session.color,
                    location: (_a = session === null || session === void 0 ? void 0 : session.location) === null || _a === void 0 ? void 0 : _a.id,
                    sessionType: (_b = session === null || session === void 0 ? void 0 : session.sessionType) === null || _b === void 0 ? void 0 : _b.id,
                    moderator: (_c = session === null || session === void 0 ? void 0 : session.moderator) === null || _c === void 0 ? void 0 : _c.id,
                    is_deleted: session === null || session === void 0 ? void 0 : session.is_deleted,
                    publishVideo: session === null || session === void 0 ? void 0 : session.publishVideo,
                    speakers: (_d = session === null || session === void 0 ? void 0 : session.speakers) === null || _d === void 0 ? void 0 : _d.map((speaker) => {
                        return speaker === null || speaker === void 0 ? void 0 : speaker.id;
                    }),
                };
            });
            const dataObj = {
                id: agenda.id,
                title: agenda.title,
                description: agenda.description,
                major_event: agenda.major_event,
                image: agenda.image,
                date: agenda.date,
                time: agenda.time,
                published: agenda.published,
                color: agenda.color,
                is_deleted: agenda.is_deleted,
                isFeatured: agenda.isFeatured,
                banner: agenda.banner,
                image2: agenda.image2,
                sponsers: agenda === null || agenda === void 0 ? void 0 : agenda.sponsers,
                agendaType: (_b = agenda === null || agenda === void 0 ? void 0 : agenda.agendaType) === null || _b === void 0 ? void 0 : _b.id,
                location: (_c = agenda === null || agenda === void 0 ? void 0 : agenda.location) === null || _c === void 0 ? void 0 : _c.id,
                sessions: sessionArray,
                logo: agenda === null || agenda === void 0 ? void 0 : agenda.logo,
                priority: agenda.priority,
                viewDetails: agenda.viewDetails,
                image3: agenda.image3,
                banner2: agenda === null || agenda === void 0 ? void 0 : agenda.banner2,
                viewAgenda: agenda === null || agenda === void 0 ? void 0 : agenda.viewAgenda,
                websiteLink: agenda === null || agenda === void 0 ? void 0 : agenda.websiteLink,
                viewWebsite: agenda === null || agenda === void 0 ? void 0 : agenda.viewWebsite,
                subHeading: agenda === null || agenda === void 0 ? void 0 : agenda.subHeading,
                image4: agenda === null || agenda === void 0 ? void 0 : agenda.image4,
                textColor: agenda === null || agenda === void 0 ? void 0 : agenda.textColor,
                permaLink: agenda === null || agenda === void 0 ? void 0 : agenda.permaLink,
                metaTitle: agenda === null || agenda === void 0 ? void 0 : agenda.metaTitle,
                metaDescription: agenda === null || agenda === void 0 ? void 0 : agenda.metaDescription,
                showFilter: agenda === null || agenda === void 0 ? void 0 : agenda.showFilter,
                isPrivateEvent: agenda === null || agenda === void 0 ? void 0 : agenda.isPrivateEvent,
                mobileBanner: agenda === null || agenda === void 0 ? void 0 : agenda.mobileBanner,
                bannerLogo: agenda === null || agenda === void 0 ? void 0 : agenda.bannerLogo,
                agendaMobileBanner: agenda === null || agenda === void 0 ? void 0 : agenda.agendaMobileBanner,
                publishVideo: agenda === null || agenda === void 0 ? void 0 : agenda.publishVideo,
                mobile_app_image: agenda === null || agenda === void 0 ? void 0 : agenda.mobile_app_image,
                mobile_app_mobile_banner: agenda === null || agenda === void 0 ? void 0 : agenda.mobile_app_mobile_banner,
                mobile_app_agenda_mobile_banner: agenda === null || agenda === void 0 ? void 0 : agenda.mobile_app_agenda_mobile_banner,
                privateSendEmail: agenda === null || agenda === void 0 ? void 0 : agenda.privateSendEmail,
                privateTitle: agenda === null || agenda === void 0 ? void 0 : agenda.privateTitle,
                privateSubTitle: agenda === null || agenda === void 0 ? void 0 : agenda.privateSubTitle,
                privateShowSector: agenda === null || agenda === void 0 ? void 0 : agenda.privateShowSector,
                privateShowAgenda: agenda === null || agenda === void 0 ? void 0 : agenda.privateShowAgenda,
                iframe: agenda === null || agenda === void 0 ? void 0 : agenda.iframe,
                showIframe: agenda === null || agenda === void 0 ? void 0 : agenda.showIframe,
            };
            return new dataResponse_dto_1.DataResponseDto(dataObj, "Agenda details fetched successfully");
        }
        catch (err) {
            console.log(err);
            if ((_d = err.details) === null || _d === void 0 ? void 0 : _d.errors) {
                return new dataResponse_dto_1.DataResponseDto(null, false, err.details.errors);
            }
            else {
                if ((_e = err.details) === null || _e === void 0 ? void 0 : _e.errors) {
                    return new dataResponse_dto_1.DataResponseDto(null, false, err.details.errors);
                }
                else {
                    return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
                }
            }
        }
    },
    async getAgendaByDate(params) {
        try {
            const { date, page, pageSize, location, id, search, speaker, isSessionFilter, isSessionSearch, filters } = params;
            let limit = Number(pageSize) || 1000;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const queryOptions = {
                where: {
                    ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                    is_deleted: false,
                    published: true,
                    ...(search &&
                        search.trim() &&
                        !isSessionSearch && { title: { $containsi: search } }),
                    ...(date && {
                        date: {
                            $gte: (0, moment_1.default)(date).startOf("day").toISOString(),
                            $lte: (0, moment_1.default)(date).endOf("day").toISOString(),
                        },
                    }),
                    ...(id && { id: id }),
                    ...(location && { location: Number(location) }),
                },
                populate: {
                    sponsers: true,
                    agendaType: true,
                    location: true,
                    agenda_sessions: {
                        where: {
                            ...(speaker && {
                                speakers: { $in: [Number(speaker)] },
                            }),
                            ...(search &&
                                search.trim() &&
                                isSessionSearch && {
                                title: { $containsi: search },
                            }),
                            ...((search === null || search === void 0 ? void 0 : search.trim()) && { title: { $containsi: search } }),
                        },
                        populate: {
                            speakers: true,
                            location: true,
                            sessionType: true,
                            moderator: true,
                        },
                    },
                },
                offset,
                limit: limit,
                orderBy: [{ date: "ASC" }, { priority: "ASC" }],
            };
            const agendas = await strapi
                .query("api::agenda.agenda")
                .findMany(queryOptions);
            const totalDataCount = await strapi
                .query("api::agenda.agenda")
                .count(queryOptions);
            const pageCount = Math.ceil(Number(totalDataCount) / limit);
            let pagination = {
                page: Number(pageNumber),
                pageSize: Number(limit),
                pageCount,
                total: totalDataCount,
            };
            let filteredAgenda = agendas;
            if (isSessionFilter) {
                filteredAgenda = agendas === null || agendas === void 0 ? void 0 : agendas.filter((agenda) => {
                    var _a;
                    return ((_a = agenda === null || agenda === void 0 ? void 0 : agenda.agenda_sessions) === null || _a === void 0 ? void 0 : _a.length) > 0;
                });
            }
            const groupedAgendas = {};
            filteredAgenda.forEach((agenda) => {
                const date = agenda.date;
                const formattedDate = (0, moment_1.default)(date).format("YYYY-MM-DD");
                if (!groupedAgendas[formattedDate]) {
                    groupedAgendas[formattedDate] = [];
                }
                groupedAgendas[formattedDate].push(agenda);
            });
            const result = Object.keys(groupedAgendas).map((date) => {
                var _a, _b;
                const sortedAgendas = groupedAgendas[date]
                    .sort((a, b) => {
                    const timeA = new Date(`2024-01-01T${a.time}Z`);
                    const timeB = new Date(`2024-01-01T${b.time}Z`);
                    return timeA.getTime() - timeB.getTime();
                });
                // sort sessions
                for (const element of sortedAgendas) {
                    if ((_a = element === null || element === void 0 ? void 0 : element.agenda_sessions) === null || _a === void 0 ? void 0 : _a.length) {
                        const sortedSessionData = (_b = element === null || element === void 0 ? void 0 : element.agenda_sessions) === null || _b === void 0 ? void 0 : _b.sort((a, b) => {
                            const dateA = a.date ? new Date(a.date) : new Date(0); // Use epoch if date is null
                            const dateB = b.date ? new Date(b.date) : new Date(0);
                            // Compare by date first
                            if (dateA.getTime() !== dateB.getTime()) {
                                return dateA.getTime() - dateB.getTime();
                            }
                            // If dates are the same, compare by toTime
                            function parseTime(timeStr) {
                                const [hours, minutes] = timeStr.split(':').map(Number);
                                const date = new Date();
                                date.setHours(hours, minutes, 0, 0);
                                return date;
                            }
                            const timeA = parseTime(a.fromTime);
                            const timeB = parseTime(b.fromTime);
                            return timeA.getTime() - timeB.getTime();
                        });
                        element.agenda_sessions = sortedSessionData;
                    }
                }
                return {
                    date: (0, moment_1.default)(date).format("YYYY-MM-DD"),
                    agendas: sortedAgendas,
                };
            });
            return {
                data: result,
                status: true,
                pagination,
                message: "Agenda details fetched successfully"
            };
        }
        catch (err) {
            console.log(err);
            return new dataResponse_dto_1.DataResponseDto(null, `Something went wrong in our end please contact system administrator`);
        }
    },
    async createAgenda(userId, body) {
        var _a;
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                var _a;
                const existingAgenda = await strapi.db.query("api::agenda.agenda").findOne({
                    where: { permaLink: (_a = body === null || body === void 0 ? void 0 : body.permaLink) === null || _a === void 0 ? void 0 : _a.trim(), is_deleted: false },
                });
                if (existingAgenda) {
                    return new dataResponse_dto_1.DataResponseDto(null, "perma link already exists");
                }
                const agenda = await strapi.entityService.create("api::agenda.agenda", {
                    data: {
                        title: body.title,
                        metaTitle: body === null || body === void 0 ? void 0 : body.metaTitle,
                        metaDescription: body === null || body === void 0 ? void 0 : body.metaDescription,
                        permaLink: body === null || body === void 0 ? void 0 : body.permaLink,
                        description: body.description,
                        agendaType: body.invitationType,
                        location: body.location,
                        image: body.image,
                        date: body.date,
                        color: body.color,
                        published: true,
                        publishedAt: new Date(),
                        time: body.time,
                        sponsers: body.sponsers,
                        isFeatured: body.isFeatured,
                        is_deleted: false,
                        banner: body.banner,
                        image2: body.image2,
                        major_event: body.major_event,
                        priority: body.priority,
                        viewDetails: body.viewDetails,
                        image3: body.image3,
                        logo: body.logo,
                        banner2: body.banner2,
                        viewAgenda: body === null || body === void 0 ? void 0 : body.viewAgenda,
                        viewWebsite: body === null || body === void 0 ? void 0 : body.viewWebsite,
                        websiteLink: body === null || body === void 0 ? void 0 : body.websiteLink,
                        subHeading: body === null || body === void 0 ? void 0 : body.subHeading,
                        image4: body === null || body === void 0 ? void 0 : body.image4,
                        textColor: body === null || body === void 0 ? void 0 : body.textColor,
                        showFilter: body === null || body === void 0 ? void 0 : body.showFilter,
                        createdBy: userId,
                        creator: body === null || body === void 0 ? void 0 : body.userId,
                        isPrivateEvent: body === null || body === void 0 ? void 0 : body.isPrivateEvent,
                        mobileBanner: body === null || body === void 0 ? void 0 : body.mobileBanner,
                        bannerLogo: body === null || body === void 0 ? void 0 : body.bannerLogo,
                        agendaMobileBanner: body === null || body === void 0 ? void 0 : body.agendaMobileBanner,
                        publishVideo: body === null || body === void 0 ? void 0 : body.publishVideo,
                        mobile_app_image: body === null || body === void 0 ? void 0 : body.mobile_app_image,
                        mobile_app_mobile_banner: body === null || body === void 0 ? void 0 : body.mobile_app_mobile_banner,
                        mobile_app_agenda_mobile_banner: body === null || body === void 0 ? void 0 : body.mobile_app_agenda_mobile_banner,
                        privateSendEmail: body === null || body === void 0 ? void 0 : body.privateSendEmail,
                        privateTitle: body === null || body === void 0 ? void 0 : body.privateTitle,
                        privateSubTitle: body === null || body === void 0 ? void 0 : body.privateSubTitle,
                        privateShowSector: body === null || body === void 0 ? void 0 : body.privateShowSector,
                        privateShowAgenda: body === null || body === void 0 ? void 0 : body.privateShowAgenda,
                        iframe: body === null || body === void 0 ? void 0 : body.iframe,
                        showIframe: body === null || body === void 0 ? void 0 : body.showIframe,
                    },
                    transacting: trx,
                });
                if (body.sessions && body.sessions.length > 0) {
                    for (const element of body.sessions) {
                        await strapi.entityService.create("api::agenda-session.agenda-session", {
                            data: {
                                day: element.day,
                                video: element.video,
                                date: element.date,
                                image: element.image,
                                title: element.title,
                                description: element.description,
                                isBreak: element.isBreak,
                                agenda: agenda.id,
                                speakers: element.speakers,
                                moderator: element.moderator,
                                fromTime: element.fromTime,
                                toTime: element.toTime,
                                location: element.location,
                                sessionType: element.sessionType,
                                color: element.color,
                                publishedAt: new Date(),
                                is_deleted: false,
                                creator: body === null || body === void 0 ? void 0 : body.userId,
                                modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                                publishVideo: element === null || element === void 0 ? void 0 : element.publishVideo
                            },
                            transacting: trx,
                        });
                    }
                }
                return new dataResponse_dto_1.DataResponseDto(agenda, "Agenda created successfully");
            });
            return result;
        }
        catch (err) {
            console.log(err);
            if ((_a = err.details) === null || _a === void 0 ? void 0 : _a.errors) {
                return new dataResponse_dto_1.DataResponseDto(null, false, err.details.errors);
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
            }
        }
    },
    async updateAgenda(id, body, userId) {
        var _a;
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                var _a;
                const existingAgenda = await strapi.db.query("api::agenda.agenda").findMany({
                    where: {
                        permaLink: (_a = body === null || body === void 0 ? void 0 : body.permaLink) === null || _a === void 0 ? void 0 : _a.trim(),
                        is_deleted: false,
                        id: {
                            $ne: id
                        }
                    },
                });
                if (existingAgenda.length) {
                    return new dataResponse_dto_1.DataResponseDto(null, "perma link already exists");
                }
                const agenda = await strapi.db.query("api::agenda.agenda").update({
                    where: { id, is_deleted: false },
                    data: {
                        title: body.title,
                        metaTitle: body === null || body === void 0 ? void 0 : body.metaTitle,
                        metaDescription: body === null || body === void 0 ? void 0 : body.metaDescription,
                        permaLink: body === null || body === void 0 ? void 0 : body.permaLink,
                        description: body.description,
                        agendaType: body.invitationType,
                        location: body.location,
                        image: body.image,
                        date: body.date,
                        color: body.color,
                        published: body.published,
                        time: body.time,
                        sponsers: body.sponsers,
                        isFeatured: body.isFeatured,
                        banner: body.banner,
                        image2: body.image2,
                        major_event: body.major_event,
                        priority: body.priority,
                        viewDetails: body.viewDetails,
                        image3: body.image3,
                        logo: body.logo,
                        banner2: body.banner2,
                        viewAgenda: body === null || body === void 0 ? void 0 : body.viewAgenda,
                        viewWebsite: body === null || body === void 0 ? void 0 : body.viewWebsite,
                        websiteLink: body === null || body === void 0 ? void 0 : body.websiteLink,
                        subHeading: body === null || body === void 0 ? void 0 : body.subHeading,
                        image4: body === null || body === void 0 ? void 0 : body.image4,
                        textColor: body === null || body === void 0 ? void 0 : body.textColor,
                        showFilter: body === null || body === void 0 ? void 0 : body.showFilter,
                        updatedBy: userId,
                        modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                        isPrivateEvent: body === null || body === void 0 ? void 0 : body.isPrivateEvent,
                        mobileBanner: body === null || body === void 0 ? void 0 : body.mobileBanner,
                        bannerLogo: body === null || body === void 0 ? void 0 : body.bannerLogo,
                        agendaMobileBanner: body === null || body === void 0 ? void 0 : body.agendaMobileBanner,
                        publishVideo: body === null || body === void 0 ? void 0 : body.publishVideo,
                        mobile_app_image: body === null || body === void 0 ? void 0 : body.mobile_app_image,
                        mobile_app_mobile_banner: body === null || body === void 0 ? void 0 : body.mobile_app_mobile_banner,
                        mobile_app_agenda_mobile_banner: body === null || body === void 0 ? void 0 : body.mobile_app_agenda_mobile_banner,
                        privateSendEmail: body === null || body === void 0 ? void 0 : body.privateSendEmail,
                        privateTitle: body === null || body === void 0 ? void 0 : body.privateTitle,
                        privateSubTitle: body === null || body === void 0 ? void 0 : body.privateSubTitle,
                        privateShowSector: body === null || body === void 0 ? void 0 : body.privateShowSector,
                        privateShowAgenda: body === null || body === void 0 ? void 0 : body.privateShowAgenda,
                        iframe: body === null || body === void 0 ? void 0 : body.iframe,
                        showIframe: body === null || body === void 0 ? void 0 : body.showIframe,
                    },
                });
                if (!agenda)
                    return new dataResponse_dto_1.DataResponseDto(null, "Agenda not found");
                const sessionIds = await strapi.db
                    .query("api::agenda-session.agenda-session")
                    .findMany({
                    select: ["id"],
                    where: { agenda: id },
                });
                if (sessionIds.length > 0) {
                    await strapi.db
                        .query("api::agenda-session.agenda-session")
                        .deleteMany({
                        where: { id: { $in: sessionIds.map((session) => session.id) } },
                    });
                }
                if (body.sessions && body.sessions.length > 0) {
                    for (const element of body.sessions) {
                        await strapi.entityService.create("api::agenda-session.agenda-session", {
                            data: {
                                day: element.day,
                                video: element.video,
                                date: element.date,
                                image: element.image,
                                title: element.title,
                                description: element.description,
                                isBreak: element.isBreak,
                                agenda: agenda.id,
                                speakers: element.speakers,
                                moderator: element.moderator,
                                fromTime: element.fromTime,
                                toTime: element.toTime,
                                location: element.location,
                                sessionType: element.sessionType,
                                color: element.color,
                                publishedAt: new Date(),
                                is_deleted: false,
                                modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                                publishVideo: element === null || element === void 0 ? void 0 : element.publishVideo
                            },
                            transacting: trx,
                        });
                    }
                }
                return new dataResponse_dto_1.DataResponseDto(agenda, "Agenda updated successfully");
            });
            return result;
        }
        catch (err) {
            console.log(err);
            if ((_a = err.details) === null || _a === void 0 ? void 0 : _a.errors) {
                return new dataResponse_dto_1.DataResponseDto(null, false, err.details.errors);
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
            }
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db.query("api::agenda.agenda").update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                    agenda_sessions: [],
                },
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "Agenda deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (err) {
            console.log(err);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async globalSearch(params) {
        try {
            const { page, pageSize, search, agendaFilters, speakerFilters, sessionFilters } = params;
            let limit = Number(pageSize) || 1000;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            // agenda
            const agendaWhereCase = {
                is_deleted: false,
                ...(agendaFilters && Object.keys(agendaFilters).length > 0 && { ...agendaFilters }),
                published: true,
                ...(search &&
                    search.trim().length && {
                    $or: [{ title: { $containsi: search } }],
                }),
            };
            const agendas = await strapi.db.query("api::agenda.agenda").findMany({
                where: agendaWhereCase,
                populate: {
                    agendaType: true,
                    location: true
                },
                offset,
                limit: limit,
                orderBy: { id: "desc" },
            });
            const totalAgendaDataCount = await strapi
                .query("api::agenda.agenda")
                .count({ where: agendaWhereCase });
            const agendaPageCount = Math.ceil(totalAgendaDataCount / limit);
            // sessions
            const sessionWhereCase = {
                is_deleted: false,
                ...(sessionFilters && Object.keys(sessionFilters).length > 0 && { ...sessionFilters }),
                published: true,
                ...(search &&
                    search.trim().length && {
                    $or: [{ title: { $containsi: search } }],
                }),
            };
            const sessions = await strapi.db.query("api::agenda-session.agenda-session").findMany({
                where: sessionWhereCase,
                populate: {
                    moderator: true,
                    location: true,
                    sessionType: true,
                    speakers: true,
                },
                offset,
                limit: limit,
                orderBy: { id: "desc" },
            });
            const totalSessionDataCount = await strapi
                .query("api::agenda-session.agenda-session")
                .count({ where: sessionWhereCase });
            const sessionPageCount = Math.ceil(totalSessionDataCount / limit);
            // speaker
            const speakerWhereCase = {
                is_deleted: false,
                ...(speakerFilters && Object.keys(speakerFilters).length > 0 && { ...speakerFilters }),
                published: true,
                ...(search &&
                    search.trim().length && {
                    $or: [
                        { firstName: { $containsi: search } },
                        { lastName: { $containsi: search } },
                        {
                            $and: [
                                { firstName: { $containsi: search.split(' ')[0] } },
                                { lastName: { $containsi: search.split(' ')[1] } },
                            ],
                        },
                    ],
                }),
            };
            const speakers = await strapi.db.query("api::speaker.speaker").findMany({
                where: speakerWhereCase,
                offset,
                limit: limit,
                orderBy: { id: "desc" },
            });
            const totalSpeakers = await strapi
                .query("api::speaker.speaker")
                .count({ where: speakerWhereCase });
            const speakerPageCount = Math.ceil(totalSpeakers / limit);
            const res = {
                agendas: {
                    data: agendas,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount: agendaPageCount,
                        total: totalAgendaDataCount,
                    }
                },
                sessions: {
                    data: sessions,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount: sessionPageCount,
                        total: totalSessionDataCount,
                    }
                },
                speakers: {
                    data: speakers,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount: speakerPageCount,
                        total: totalSpeakers,
                    }
                }
            };
            return new dataResponse_dto_1.DataResponseDto(res, "Data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllInitialAppData(params) {
        try {
            const { agendaLimit, sessionLimit, bannerLimit, speakerLimit, entertainmentLimit, partnersLimit } = params;
            // fetch latest 6 featured events
            const agenda_limit = Number(agendaLimit) || 6;
            const featuredAgendas = await strapi.db.query("api::agenda.agenda").findMany({
                where: { is_deleted: false, major_event: true, published: true, isPrivateEvent: false },
                populate: {
                    location: true
                },
                orderBy: { priority: "asc" },
                limit: Number(agenda_limit),
                select: ["id", "image", "mobile_app_image", "permaLink", "image4", "mobile_app_image", "mobile_app_mobile_banner", "mobile_app_agenda_mobile_banner"]
            });
            // fetch sessions
            const sessions = await strapi.db.query("api::agenda-session.agenda-session").findMany({
                where: {
                    is_deleted: false,
                    agenda: {
                        $notNull: true,
                    },
                },
                orderBy: [
                    { agenda: { date: "asc" } },
                    { fromTime: "asc" } // Changed from toTime to fromTime for consistency
                ],
                select: ["id", "title", "description", "fromTime", "toTime", "day", "date", "color", "image", "video", "published", "publishVideo", "isBreak"],
                populate: {
                    agenda: {
                        select: ["id", "title", "date", "logo", "mobileBanner", "agendaMobileBanner", "color", "image4", "mobile_app_image", "mobile_app_mobile_banner", "mobile_app_agenda_mobile_banner"]
                    },
                    sessionType: {
                        select: ["name", "icon"]
                    },
                    moderator: {
                        select: ["firstName", "lastName", "photoUrl", "companyName", "designation", "mobile_app_photo_url"]
                    },
                    location: {
                        select: ["name"]
                    },
                    speakers: {
                        select: ["firstName", "lastName", "photoUrl", "companyName", "designation", "mobile_app_photo_url"]
                    },
                }
            });
            function parseTime(timeStr) {
                const [hours, minutes] = timeStr.split(':').map(Number);
                const dubaiDate = new Date(new Date().toLocaleString('en-US', { timeZone: 'Asia/Dubai' }));
                dubaiDate.setHours(hours, minutes, 0, 0);
                return dubaiDate;
            }
            const dubaiNow = new Date(new Date().toLocaleString('en-US', { timeZone: 'Asia/Dubai' }));
            const currentTime = dubaiNow;
            const currentDate = new Date(dubaiNow);
            currentDate.setHours(0, 0, 0, 0);
            const upcomingSessions = sessions
                .filter((item) => {
                var _a;
                if (!((_a = item.agenda) === null || _a === void 0 ? void 0 : _a.date))
                    return false;
                const sessionDate = new Date(new Date(item.agenda.date).toLocaleString('en-US', { timeZone: 'Asia/Dubai' }));
                sessionDate.setHours(0, 0, 0, 0);
                // If session is today, check if the time hasn't passed yet
                if (sessionDate.getTime() === currentDate.getTime()) {
                    const sessionTime = parseTime(item.fromTime);
                    return sessionTime > currentTime;
                }
                // Include future dates
                return sessionDate > currentDate;
            })
                .sort((a, b) => {
                // Compare agenda dates first
                const dateA = new Date(new Date(a.agenda.date).toLocaleString('en-US', { timeZone: 'Asia/Dubai' }));
                const dateB = new Date(new Date(b.agenda.date).toLocaleString('en-US', { timeZone: 'Asia/Dubai' }));
                if (dateA.getTime() !== dateB.getTime()) {
                    return dateA.getTime() - dateB.getTime();
                }
                // If same date, compare by fromTime
                const timeA = parseTime(a.fromTime);
                const timeB = parseTime(b.fromTime);
                return timeA.getTime() - timeB.getTime();
            })
                .slice(0, 6);
            // fetch latest banners
            const banner_limit = Number(bannerLimit) || 6;
            const banners = await strapi.db.query("api::banner.banner").findMany({
                where: { is_deleted: false },
                orderBy: { position: "asc" },
                limit: Number(banner_limit),
                select: ["image", "position", "mobile_app_image"]
            });
            // fetch speakers
            const speaker_limit = Number(speakerLimit) || 9;
            const speakers = await strapi.db.query("api::speaker.speaker").findMany({
                where: {
                    is_deleted: false,
                    published: true
                },
                populate: {
                    agenda_sessions: {
                        populate: {
                            agenda: {
                                select: ["id", "title", "description", "color"]
                            }
                        },
                        select: ["id", "isBreak"]
                    }
                },
                limit: Number(speaker_limit),
                orderBy: { priority: "asc" },
                select: ["firstName", "lastName", "companyName", "designation", "photoUrl", "mobile_app_photo_url", "bio", "mobile_app_photo_url"]
            });
            // locations
            const locations = await strapi.db.query("api::venue.venue").findMany({
                where: {
                    is_deleted: false,
                    showMap: true
                }
            });
            // entertainment banner images
            const entertainment_limit = Number(entertainmentLimit) || 3;
            const entertainments = await strapi.db.query("api::entertainment.entertainment").findMany({
                where: { is_deleted: false },
                limit: Number(entertainment_limit),
                select: ["banner", "name"]
            });
            // partners website images by categories
            const partners_limit = Number(partnersLimit) || 5;
            const partners = await strapi.db.query("api::sponsors-category.sponsors-category").findMany({
                where: {
                    showWebsite: true,
                },
                populate: {
                    sponsors: {
                        where: {
                            is_deleted: false,
                            websiteImage: {
                                $notNull: true
                            }
                        },
                        orderBy: { priority: "asc" },
                        select: ["priority", "websiteImage"]
                    }
                },
                orderBy: { priority: "asc" },
                select: ["value"],
                limit: Number(partners_limit)
            });
            let sessionsData = (upcomingSessions === null || upcomingSessions === void 0 ? void 0 : upcomingSessions.length) ? upcomingSessions : sessions === null || sessions === void 0 ? void 0 : sessions.slice(0, 6);
            const responseData = {
                featuredAgendas,
                sessions: sessionsData,
                banners,
                speakers,
                locations,
                entertainments,
                partners
            };
            return new dataResponse_dto_1.DataResponseDto(responseData, "Data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, `Something went wrong in our end please contact system administrator`);
        }
    }
}));
