"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const axios = require("axios");
module.exports = {
    Login: async (email, password) => {
        try {
            if (!email || !password) {
                return {
                    data: null,
                    status: false,
                    message: "Email and password are required.",
                };
            }
            const user = await strapi
                .query("plugin::users-permissions.user")
                .findOne({
                where: { email: email === null || email === void 0 ? void 0 : email.trim(), is_deleted: false },
                populate: { role: true },
            });
            if (!user) {
                return {
                    data: null,
                    status: false,
                    message: "User not found.",
                };
            }
            if (user === null || user === void 0 ? void 0 : user.blocked) {
                return {
                    data: null,
                    status: false,
                    message: "Your account has been blocked. Please contact support.",
                };
            }
            const isPasswordValid = await strapi
                .service("plugin::users-permissions.user")
                .validatePassword(password, user === null || user === void 0 ? void 0 : user.password);
            if (!isPasswordValid) {
                return {
                    data: null,
                    status: false,
                    message: "Invalid email or password.",
                };
            }
            const jwtToken = await strapi
                .service("plugin::users-permissions.jwt")
                .issue({ id: user.id });
            return { jwt: jwtToken, user, message: "successfully login" };
        }
        catch (err) {
            strapi.log.error(`Login error: ${err.message}`);
            throw new Error(err.message);
        }
    },
    register: async (body) => {
        try {
            const { password, username, email, role, firstName, lastName, phone, country_code, session_access, photo, blocked, user_role, company, designation, } = body;
            const hashPassword = await strapi
                .service("admin::auth")
                .hashPassword(password);
            let obj = {
                data: {
                    username: username,
                    email: email,
                    password: hashPassword,
                    role: role,
                    user_role: user_role,
                    blocked: blocked,
                    confirmed: true,
                    firstName,
                    lastName,
                    phone,
                    country_code,
                    session_access,
                    photo,
                    company,
                    designation,
                },
            };
            const user = await strapi.db
                .query("plugin::users-permissions.user")
                .create(obj);
            const { password: resPass, ...rest } = user;
            const jwtToken = await strapi
                .service("plugin::users-permissions.jwt")
                .issue({ id: user.id });
            if (user) {
                return {
                    status: true,
                    data: { ...rest, token: jwtToken },
                    message: "User registered successfully",
                };
            }
            else {
                return {
                    status: false,
                    data: {},
                    message: "User registered Failed",
                };
            }
        }
        catch (err) {
            throw new Error(err.message);
        }
    },
    userLogin: async (body) => {
        try {
            const { ticketNumber, lastName, fcmToken, webLogin } = body;
            if (!ticketNumber || !lastName) {
                return new dataResponse_dto_1.DataResponseDto(null, "Provide all required fields");
            }
            ;
            let user = await strapi
                .query("api::ticket-master.ticket-master")
                .findOne({
                where: {
                    ticketNumber: {
                        $eqi: ticketNumber === null || ticketNumber === void 0 ? void 0 : ticketNumber.trim(),
                    },
                    lastName: {
                        $eqi: lastName,
                    },
                    status: "issued",
                    is_deleted: false,
                },
                populate: {
                    ticket_id: true
                },
            });
            if (!user) {
                return new dataResponse_dto_1.DataResponseDto(null, "Invalid credentials");
            }
            if (!webLogin && user.ticket_id.id_short === "general_admission") {
                return new dataResponse_dto_1.DataResponseDto(null, "Your ticket does not include ADFW mobile app access");
            }
            ;
            let sessionData;
            if (!webLogin) {
                let existingSessions = await strapi
                    .query("api::app-session-log.app-session-log")
                    .findOne({
                    where: {
                        ticket_number: ticketNumber === null || ticketNumber === void 0 ? void 0 : ticketNumber.trim(),
                        is_active: true,
                        device_id: body === null || body === void 0 ? void 0 : body.device_id,
                    }
                });
                if (existingSessions) {
                    await strapi.db.query("api::app-session-log.app-session-log").update({
                        where: { id: existingSessions.id },
                        data: {
                            is_active: false,
                            logout_time: new Date()
                        },
                    });
                }
                // app session log
                sessionData = await strapi.entityService.create("api::app-session-log.app-session-log", {
                    data: {
                        ticket_number: ticketNumber.trim(),
                        is_active: true,
                        ticket_id: user === null || user === void 0 ? void 0 : user.id,
                        login_time: new Date(),
                        logout_time: null,
                        device_id: body === null || body === void 0 ? void 0 : body.device_id,
                        details: body === null || body === void 0 ? void 0 : body.details,
                        publishedAt: new Date()
                    }
                });
            }
            const token = await strapi
                .service("plugin::users-permissions.jwt")
                .issue({ ticketNumber: ticketNumber });
            if (fcmToken) {
                let existingFcmToken = (user === null || user === void 0 ? void 0 : user.fcmToken) || [];
                existingFcmToken.push(fcmToken);
                existingFcmToken = [...new Set(existingFcmToken)];
                user = await strapi.db.query("api::ticket-master.ticket-master").update({
                    where: { id: user === null || user === void 0 ? void 0 : user.id },
                    data: { fcmToken: existingFcmToken },
                    populate: {
                        ticket_id: true
                    }
                });
            }
            return new dataResponse_dto_1.DataResponseDto({ user, token, sessionId: sessionData === null || sessionData === void 0 ? void 0 : sessionData.id }, "Logged in successfully");
        }
        catch (err) {
            console.log("err", err);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    userLogout: async (ticketNumber, fcmToken, sessionId) => {
        var _a;
        try {
            const user = await strapi.db
                .query("api::ticket-master.ticket-master")
                .findOne({
                where: {
                    ticketNumber: ticketNumber === null || ticketNumber === void 0 ? void 0 : ticketNumber.trim(),
                    status: "issued",
                    is_deleted: false,
                }
            });
            if (!user) {
                return new dataResponse_dto_1.DataResponseDto(null, "No user found");
            }
            // update fcm token 
            if (fcmToken && (user === null || user === void 0 ? void 0 : user.fcmToken)) {
                const updatedFcmToken = (_a = user === null || user === void 0 ? void 0 : user.fcmToken) === null || _a === void 0 ? void 0 : _a.filter(token => token !== fcmToken);
                await strapi.db.query("api::ticket-master.ticket-master").update({
                    where: { id: user === null || user === void 0 ? void 0 : user.id },
                    data: { fcmToken: updatedFcmToken },
                });
            }
            // update the log 
            if (sessionId) {
                await strapi.db.query("api::app-session-log.app-session-log").update({
                    where: { id: sessionId },
                    data: {
                        is_active: false,
                        logout_time: new Date()
                    },
                });
            }
            return new dataResponse_dto_1.DataResponseDto("Success", "User logout successfully");
        }
        catch (error) {
            console.log("err", error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    updateUserDetails: async (id, body) => {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        try {
            const { firstName, lastName, email, company, sector, designation, nationality, residanceCountry, countryCode, mobile, allowEmail, photo, emirate, ticketId, day, notes, badgeCategory, isTermsAgreed, openingNight, openingCeremony, isConfirmed, tag, } = body;
            const existingData = await strapi
                .query("api::ticket-master.ticket-master")
                .findOne({
                where: { id, is_deleted: false, status: {
                        $notIn: ["refunded", "cancelled", "deleted"]
                    } },
                populate: {
                    ticket_id: true,
                },
            });
            if (!existingData) {
                return new dataResponse_dto_1.DataResponseDto(null, "User ticket not found");
            }
            let dataObj = {
                firstName: firstName || existingData.firstName,
                lastName: lastName || existingData.lastName,
                email: email || existingData.email,
                company: company || existingData.company,
                sector: sector || existingData.sector,
                designation: designation || existingData.designation,
                nationality: nationality || existingData.nationality,
                residanceCountry: residanceCountry || existingData.residanceCountry,
                countryCode: countryCode,
                mobile: mobile,
                allowEmail: allowEmail || existingData.allowEmail,
                emirate: emirate || existingData.emirate,
                ticket_id: ticketId || existingData.ticket_id,
                day: day || existingData.day,
                notes: notes || existingData.notes || "",
                badgeCategory: badgeCategory || existingData.badgeCategory,
                photo: photo || existingData.photo,
                dayShort: ((_a = body === null || body === void 0 ? void 0 : body.day) === null || _a === void 0 ? void 0 : _a.id_short) || existingData.dayShort,
                isTermsAgreed: isTermsAgreed || existingData.isTermsAgreed,
                openingCeremony: openingCeremony || existingData.openingCeremony,
                openingNight: openingNight || existingData.openingNight,
                isConfirmed: isConfirmed || existingData.isConfirmed,
                tag: tag
            };
            const user = await strapi
                .query("api::ticket-master.ticket-master")
                .update({
                where: { id },
                data: dataObj,
                populate: {
                    ticket_id: true
                }
            });
            let externalRegister;
            if ((existingData === null || existingData === void 0 ? void 0 : existingData.status) === "issued") {
                let payment = existingData.paymentStatus === "Free" ? "Foc" : "Paid";
                let URL = process.env.EVENTO_VISITOR_INSERT;
                let reqObj = {
                    OrderNumber: body.orderNumber || existingData.orderNumber,
                    VisitorNumber: body.ticketNumber || existingData.ticketNumber,
                    FirstName: body.firstName || existingData.firstName,
                    LastName: body.lastName || existingData.lastName,
                    Email: body.email || existingData.email,
                    Company: body.company || existingData.company,
                    Sector: body.sector || existingData.sector,
                    JobTitle: body.designation || existingData.designation,
                    Nationality: body.nationality || existingData.nationality,
                    Country: body.residanceCountry || existingData.residanceCountry,
                    Emirate: body.emirate || "",
                    Photo: body.photo || existingData.photo,
                    Mobile: ((body === null || body === void 0 ? void 0 : body.countryCode) &&
                        (body === null || body === void 0 ? void 0 : body.mobile) &&
                        `${body.countryCode.replace("+", "")} ${body.mobile}`) ||
                        ((existingData === null || existingData === void 0 ? void 0 : existingData.countryCode) &&
                            (existingData === null || existingData === void 0 ? void 0 : existingData.mobile) &&
                            `${existingData.countryCode.replace("+", "")} ${existingData.mobile}`),
                    City: (body === null || body === void 0 ? void 0 : body.nationality) || existingData.nationality,
                    Quantiy: 1,
                    Payment: payment,
                    BadgeCategory: (body === null || body === void 0 ? void 0 : body.badgeCategory) || existingData.badgeCategory,
                    SingleDay: ((_b = body === null || body === void 0 ? void 0 : body.day) === null || _b === void 0 ? void 0 : _b.event) || ((_c = existingData.day) === null || _c === void 0 ? void 0 : _c.event),
                    TicketType: (body === null || body === void 0 ? void 0 : body.ticketType) || ((_d = existingData.ticket_id) === null || _d === void 0 ? void 0 : _d.id_short),
                };
                externalRegister = await axios.post(URL, reqObj, {
                    headers: {
                        Authorization: process.env.EVENTO_VISITOR_TOKEN,
                        "Content-Type": "application/json",
                    },
                });
            }
            if ((body === null || body === void 0 ? void 0 : body.sendEmail) && (user === null || user === void 0 ? void 0 : user.allowEmail)) {
                let mandrilURL = process.env.MANDRILL_SENT_MAIL_URL;
                let obj = {
                    key: process.env.MANDRILL_KEY,
                    template_name: process.env.UPGRADE_TICKET_EMAIL_TEMPLATE,
                    template_content: [],
                    message: {
                        to: [
                            {
                                email: user === null || user === void 0 ? void 0 : user.email,
                                name: user === null || user === void 0 ? void 0 : user.firstName,
                                type: "to",
                            },
                        ],
                        from_email: "donotreply@adfw.com",
                        from_name: "Abu Dhabi Finance Week",
                        global_merge_vars: [
                            {
                                name: "FIRSTNAME",
                                content: user === null || user === void 0 ? void 0 : user.firstName,
                            },
                            {
                                name: "FULLNAME",
                                content: `${user === null || user === void 0 ? void 0 : user.firstName} ${user === null || user === void 0 ? void 0 : user.lastName}`,
                            },
                            {
                                name: "EMAIL",
                                content: user === null || user === void 0 ? void 0 : user.email,
                            },
                            {
                                name: "TICKETTYPE",
                                content: ((_f = (_e = user.day) === null || _e === void 0 ? void 0 : _e.event) === null || _f === void 0 ? void 0 : _f.toLowerCase()) !== "other"
                                    ? `${(_g = user === null || user === void 0 ? void 0 : user.ticket_id) === null || _g === void 0 ? void 0 : _g.title}: ${(_h = user === null || user === void 0 ? void 0 : user.day) === null || _h === void 0 ? void 0 : _h.event}`
                                    : (_j = user === null || user === void 0 ? void 0 : user.ticket_id) === null || _j === void 0 ? void 0 : _j.title
                            },
                            {
                                name: "TICKETID",
                                content: user === null || user === void 0 ? void 0 : user.ticketNumber,
                            },
                            {
                                name: "DOWNLOADTICKET",
                                content: `${process.env.WEB_URL}ticket-download?ticketNumber=${user === null || user === void 0 ? void 0 : user.ticketNumber}`,
                            },
                            {
                                name: "APPLEWALLET",
                                content: `${process.env.SERVER_URL}apple-wallet/download-pass?ticketNumber=${user === null || user === void 0 ? void 0 : user.ticketNumber}`,
                            },
                            {
                                name: "GOOGLEWALLET",
                                content: `${process.env.SERVER_URL}wallet/google-wallet?ticketNumber=${user === null || user === void 0 ? void 0 : user.ticketNumber}`,
                            },
                        ],
                    },
                };
                await axios.post(mandrilURL, obj);
            }
            return new dataResponse_dto_1.DataResponseDto({ data: user, eventoData: externalRegister === null || externalRegister === void 0 ? void 0 : externalRegister.data }, "User ticket details updated successfully");
        }
        catch (err) {
            console.log(err);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    findUsers: async (params) => {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                is_deleted: false,
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search &&
                    search.trim().length && {
                    $or: [
                        { username: { $containsi: search } },
                        { email: { $containsi: search } },
                        { firstName: { $containsi: search } },
                        { lastName: { $containsi: search } },
                    ],
                }),
            };
            const users = await strapi
                .query("plugin::users-permissions.user")
                .findMany({
                where,
                populate: { role: true },
                offset,
                limit: limit,
                orderBy: { id: "desc" },
            });
            const totalUsers = await strapi
                .query("plugin::users-permissions.user")
                .count({ where });
            const pageCount = Math.ceil(totalUsers / limit);
            return new dataResponse_dto_1.DataResponseDto({
                users,
                pagination: {
                    page: Number(pageNumber),
                    pageSize: Number(limit),
                    pageCount,
                    total: totalUsers,
                },
            }, "Users fetched successfully");
        }
        catch (err) {
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    findExportData: async () => {
        try {
            const users = await strapi
                .query("plugin::users-permissions.user")
                .findMany({
                where: { is_deleted: false },
                populate: { role: true },
                orderBy: { id: "desc" },
            });
            return new dataResponse_dto_1.DataResponseDto(users, "User data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    findOneUser: async (id) => {
        try {
            const user = await strapi
                .query("plugin::users-permissions.user")
                .findOne({ where: { id }, populate: { role: true } });
            if (user) {
                return new dataResponse_dto_1.DataResponseDto(user, "User fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null, "User not found");
            }
        }
        catch (err) {
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    softDelete: async (id) => {
        try {
            const user = await strapi.db
                .query("plugin::users-permissions.user")
                .update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            if (user) {
                return new dataResponse_dto_1.DataResponseDto(user, "User deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null, "User not found");
            }
        }
        catch (err) {
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    updateUser: async (id, body) => {
        try {
            const data = {};
            if (body.username)
                data.username = body.username;
            if (body.firstName)
                data.firstName = body.firstName;
            if (body.lastName)
                data.lastName = body.lastName;
            if (body.email)
                data.email = body.email;
            if (body.company)
                data.company = body.company;
            if (body.session_access)
                data.session_access = body.session_access;
            data.blocked = body.blocked;
            if (body.designation)
                data.designation = body.designation;
            if (body.photo)
                data.photo = body.photo;
            if (body.role)
                data.role = body.role;
            if (body.user_role)
                data.user_role = body.user_role;
            if (body.phone)
                data.phone = body.phone;
            if (body.country_code)
                data.country_code = body.country_code;
            if (body.password) {
                let newPassword = await strapi
                    .service("admin::auth")
                    .hashPassword(body.password);
                data.password = newPassword;
            }
            const user = await strapi.query("plugin::users-permissions.user").update({
                where: { id },
                data,
            });
            if (user) {
                return new dataResponse_dto_1.DataResponseDto(user, "User added successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null, "Failed to create user");
            }
        }
        catch (err) {
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
};
