"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const moment_1 = __importDefault(require("moment"));
exports.default = strapi_1.factories.createCoreService("api::entertainment.entertainment", ({ strapi }) => ({
    async findAllData(params) {
        try {
            const { page, pageSize, search, filters = {} } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const { date, ...otherFilters } = filters;
            // where condition for artists
            const where = {
                is_deleted: false,
                ...(otherFilters && Object.keys(otherFilters).length > 0 && { ...otherFilters }),
                ...(date && {
                    date: {
                        $gte: (0, moment_1.default)(date).startOf("day").toISOString(),
                        $lte: (0, moment_1.default)(date).endOf("day").toISOString(),
                    },
                }),
                ...(search &&
                    search.trim() && {
                    $or: [
                        { name: { $containsi: search } }
                    ],
                }),
            };
            // fetch all entertainments
            const data = await strapi.db.query("api::entertainment.entertainment").findMany({
                where: {
                    is_deleted: false
                },
                populate: {
                    artists: {
                        where,
                        populate: {
                            location: true
                        }
                    }
                },
                offset,
                limit: limit,
                orderBy: { id: "desc" }
            });
            const entertainments = [];
            if (data === null || data === void 0 ? void 0 : data.length) {
                data.forEach((item) => {
                    var _a;
                    const groupedShows = {};
                    //  group by date
                    (_a = item === null || item === void 0 ? void 0 : item.artists) === null || _a === void 0 ? void 0 : _a.forEach((show) => {
                        const formattedDate = (0, moment_1.default)(show.date).format("YYYY-MM-DD");
                        if (!groupedShows[formattedDate]) {
                            groupedShows[formattedDate] = [];
                        }
                        groupedShows[formattedDate].push(show);
                    });
                    // Process each date group
                    const result = Object.keys(groupedShows).map((date) => {
                        const sortedShows = groupedShows[date].sort((a, b) => {
                            // Create full date-time objects for comparison
                            const dateTimeA = (0, moment_1.default)(`${a.date} ${a.time}`, "YYYY-MM-DD HH:mm");
                            const dateTimeB = (0, moment_1.default)(`${b.date} ${b.time}`, "YYYY-MM-DD HH:mm");
                            // If either date is invalid, handle appropriately
                            if (!dateTimeA.isValid() || !dateTimeB.isValid()) {
                                console.warn('Invalid date detected:', {
                                    showA: { date: a.date, time: a.time },
                                    showB: { date: b.date, time: b.time }
                                });
                                return 0; // Keep original order for invalid dates
                            }
                            // Compare the full date-time
                            return dateTimeA.valueOf() - dateTimeB.valueOf();
                        });
                        return {
                            date: (0, moment_1.default)(date).format("YYYY-MM-DD"),
                            shows: sortedShows,
                        };
                    });
                    // Sort the dates themselves
                    result.sort((a, b) => (0, moment_1.default)(a.date).valueOf() - (0, moment_1.default)(b.date).valueOf());
                    let resultObj = {
                        banner: item === null || item === void 0 ? void 0 : item.banner,
                        name: item === null || item === void 0 ? void 0 : item.name,
                        artists: result
                    };
                    entertainments.push(resultObj);
                });
            }
            const totalDataCount = await strapi
                .query("api::entertainment.entertainment")
                .count({ where: {
                    is_deleted: false
                } });
            const pageCount = Math.ceil(totalDataCount / limit);
            return new dataResponse_dto_1.DataResponseDto({
                entertainments,
                pagination: {
                    page: Number(pageNumber),
                    pageSize: Number(limit),
                    pageCount,
                    total: Number(totalDataCount),
                },
            }, "Data fetched successfully");
        }
        catch (error) {
            console.log("error", error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db
                .query("api::entertainment.entertainment")
                .update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "entertainment deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    }
}));
