"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
exports.default = strapi_1.factories.createCoreService("api::favourite.favourite", ({ strapi }) => ({
    async findFavouritesByTicketId(id, params) {
        try {
            const { page, pageSize, search, agendaId, locationId } = params;
            const limit = Number(pageSize) || 10;
            const pageNumber = Number(page) || 1;
            const offset = (pageNumber - 1) * limit;
            const results = await strapi.db.connection.raw(`
          SELECT 
            DATE(f.date) AS date,
            JSON_AGG(
              JSON_BUILD_OBJECT(
                'agenda_id', a.id,
                'agenda_title', a.title,
                'agenda_banner', a.banner,
                'agenda_logo', a.logo,
                'agenda_mobile_banner', a.mobile_banner,
                'agenda_agenda_mobile_banner', a.agenda_mobile_banner,
                'location',v.name,
               'session_id', s.id,
                'session_title', s.title,
                'session_from_time', s.from_time,
                'session_to_time', s.to_time,
                'session_video', s.video,
                'session_description',s.description,
                'session_type', (
                        SELECT COALESCE(
                          JSONB_BUILD_OBJECT(
                            'id', st.id,
                            'name', st.name,
                            'icon', st.icon
                          ),
                          NULL
                        )
                        FROM agenda_sessions_session_type_links sstl
                        JOIN agenda_session_types st ON st.id = sstl.agenda_session_type_id
                        WHERE sstl.agenda_session_id = s.id
                      ),
                      'speakers', (
                        SELECT COALESCE(
                          JSON_AGG(
                            JSONB_BUILD_OBJECT(
                              'id', sp.id,
                              'first_name', sp.first_name,
                              'last_name', sp.last_name,
                              'photo_url', sp.photo_url,
                              'designation', sp.designation,
                              'company_name',sp.company_name
                            )
                          ),
                          '[]'::json
                        )
                        FROM speakers_agenda_sessions_links sasl
                        JOIN speakers sp ON sp.id = sasl.speaker_id
                        WHERE sasl.agenda_session_id = s.id
                      ),
                      'moderators', (
                        SELECT COALESCE(
                          JSON_AGG(
                            JSONB_BUILD_OBJECT(
                              'id', sp.id,
                              'first_name', sp.first_name,
                              'last_name', sp.last_name,
                              'photo_url', sp.photo_url,
                               'designation', sp.designation,
                              'company_name',sp.company_name
                            )
                          ),
                          '[]'::json
                        )
                        FROM agenda_sessions_moderator_links asml
                        JOIN speakers sp ON sp.id = asml.speaker_id
                        WHERE asml.agenda_session_id = s.id
                      )
              )  ORDER BY s.from_time
            ) AS sessions
          FROM favourites AS f
          JOIN favourites_agenda_session_id_links AS fasl ON f.id = fasl.favourite_id
          JOIN agenda_sessions AS s ON fasl.agenda_session_id = s.id
          JOIN favourites_ticket_id_links AS ftl ON fasl.favourite_id = ftl.favourite_id
          JOIN favourites_agenda_links AS fal ON fasl.favourite_id = fal.favourite_id
          JOIN agendas AS a ON fal.agenda_id = a.id 
          JOIN agendas_location_links AS al ON a.id = al.agenda_id
          JOIN venues AS v ON al.venue_id = v.id
          WHERE ftl.ticket_master_id = ?
          AND f.date IS NOT NULL
           ${agendaId ? 'AND fal.agenda_id = ?' : ''}
           ${locationId ? 'AND v.id = ?' : ''}
           ${search ? 'AND s.title ILIKE ?' : ''}
          GROUP BY DATE(f.date)
          ORDER BY DATE(f.date)
          LIMIT ? OFFSET ?;
          `, [
                id,
                ...(agendaId ? [agendaId] : []),
                ...(locationId ? [locationId] : []),
                ...(search ? [`%${search}%`] : []),
                limit,
                offset
            ]);
            // modify data format
            const groupedData = results.rows.map(dateObj => {
                const groupedSessions = dateObj.sessions.reduce((acc, session) => {
                    if (!acc[session.agendaId]) {
                        acc[session.agendaId] = {
                            agenda_id: session.agenda_id,
                            agenda_title: session.agenda_title,
                            agenda_banner: session.agenda_banner,
                            agenda_logo: session.agenda_logo,
                            agenda_mobile_banner: session.agenda_mobile_banner,
                            agenda_agenda_mobile_banner: session.agenda_agenda_mobile_banner,
                            location: session.location,
                            sessions: []
                        };
                    }
                    acc[session.agendaId].sessions.push({
                        session_id: session.session_id,
                        session_title: session.session_title,
                        session_description: session.session_description,
                        session_from_time: session.session_from_time,
                        session_to_time: session.session_to_time,
                        session_video: session.session_video,
                        session_type: session.session_type,
                        speakers: session.speakers,
                        moderators: session.moderators,
                    });
                    return acc;
                }, {});
                return {
                    date: dateObj.date,
                    agendas: Object.values(groupedSessions)
                };
            });
            // Get total count 
            const countResult = await strapi.db.connection.raw(`
          SELECT 
            COUNT(DISTINCT DATE(f.date)) AS total_count
          FROM favourites AS f
          JOIN favourites_agenda_session_id_links AS fasl ON f.id = fasl.favourite_id
          JOIN agenda_sessions AS s ON fasl.agenda_session_id = s.id
          JOIN favourites_ticket_id_links AS ftl ON fasl.favourite_id = ftl.favourite_id
          JOIN favourites_agenda_links AS fal ON fasl.favourite_id = fal.favourite_id
          JOIN agendas AS a ON fal.agenda_id = a.id 
          JOIN agendas_location_links AS al ON a.id = al.agenda_id
          JOIN venues AS v ON al.venue_id = v.id
          WHERE ftl.ticket_master_id = ?
          AND f.date IS NOT NULL
            ${agendaId ? 'AND fal.agenda_id = ?' : ''}
            ${locationId ? 'AND v.id = ?' : ''}
            ${search ? 'AND s.title ILIKE ?' : ''}
          `, [
                id,
                ...(agendaId ? [agendaId] : []),
                ...(locationId ? [locationId] : []),
                ...(search ? [`%${search}%`] : [])
            ]);
            const totalCount = countResult.rows[0].total_count;
            const totalPages = Math.ceil(Number(totalCount) / limit);
            const response = {
                data: groupedData,
                status: true,
                message: "Favourites fetched successfully",
                meta: {
                    pagination: {
                        page: pageNumber,
                        pageSize: limit,
                        pageCount: Number(totalPages),
                        total: Number(totalCount),
                        hasNextPage: pageNumber < totalPages,
                        hasPreviousPage: pageNumber > 1
                    }
                }
            };
            return response;
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong on our end; please contact the system administrator");
        }
    },
    async addToFavourites(ticketId, sessionId, agendaId, agendaDate) {
        try {
            const favouriteData = await strapi.db.query("api::favourite.favourite").findOne({
                where: { ticket_id: ticketId, agenda_session_id: sessionId, is_deleted: false }
            });
            if (favouriteData) {
                return new dataResponse_dto_1.DataResponseDto(null, "This session already exists in your favourites");
            }
            const data = await strapi.db.query("api::favourite.favourite").create({
                data: {
                    ticket_id: ticketId,
                    agenda_session_id: sessionId,
                    agenda: agendaId,
                    date: agendaDate,
                    publishedAt: new Date(),
                },
                populate: {
                    agenda_session_id: {
                        select: ["id"]
                    }
                }
            });
            return new dataResponse_dto_1.DataResponseDto(data, "Session added to your favourites");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async removeFromFavourites(ticketId, sessionId) {
        try {
            const favouriteData = await strapi.db.query("api::favourite.favourite").findOne({
                where: { ticket_id: ticketId, agenda_session_id: sessionId, is_deleted: false }
            });
            if (!favouriteData) {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
            const deletedData = await strapi.db.query("api::favourite.favourite").delete({
                where: { ticket_id: ticketId, agenda_session_id: sessionId, is_deleted: false }
            });
            return new dataResponse_dto_1.DataResponseDto(deletedData, "Item removed from favourites");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async removeAllFavourites(ticketId) {
        try {
            // check the user exists or not
            const ticketData = await strapi.db
                .query("api::ticket-master.ticket-master")
                .findOne({
                where: { id: ticketId, is_deleted: false },
                populate: {
                    ticket_id: true,
                    adgm: true
                },
            });
            // throw error if there is no user without this id
            if (!ticketData) {
                return new dataResponse_dto_1.DataResponseDto(null, "No user found for this id");
            }
            // fetch all favourites data
            const favouriteData = await strapi.db.query("api::favourite.favourite").findMany({
                where: { ticket_id: ticketId, is_deleted: false },
                select: ["id"]
            });
            // group the favourites ids
            const favouritesIds = favouriteData.map((item) => item === null || item === void 0 ? void 0 : item.id);
            // soft delete all favourites usign the primary key id
            const deltedItems = await strapi.db.query("api::favourite.favourite").updateMany({
                where: {
                    id: {
                        $in: favouritesIds
                    }
                },
                data: { is_deleted: true }
            });
            return new dataResponse_dto_1.DataResponseDto("success", "Favourites data cleared successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db.query("api::favourite.favourite").update({
                where: { id },
                data: { is_deleted: true }
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "Item deleted from favourites successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
}));
