"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const fs = require("fs");
const Jimp = require("jimp");
const AWS = require("aws-sdk");
const moment = require("moment");
const { BlobServiceClient, StorageSharedKeyCredential } = require("@azure/storage-blob");
AWS.config.update({
    accessKeyId: process.env.S3_ACCESS_KEY,
    secretAccessKey: process.env.S3_SECRET_KEY,
    region: process.env.S3_REGION,
});
const s3 = new AWS.S3();
module.exports = {
    uploadImage: async (imageFile) => {
        try {
            if (!imageFile.files.path) {
                return new dataResponse_dto_1.DataResponseDto(null, "Failed to upload image file");
            }
            const data = await fs.promises.readFile(imageFile.files.path);
            const image = await Jimp.read(data);
            const compressedImage = await image
                .resize(Jimp.AUTO, 500)
                .quality(60)
                .getBufferAsync(Jimp.AUTO);
            const title = imageFile.files.name.split(".")[0];
            const folderName = "images";
            const fileName = moment().format("YYYY-MM-DDHHmmss") + "-" + title + ".png";
            const fileKey = folderName + "/" + fileName;
            const params = {
                Bucket: process.env.S3_BUCKET,
                Key: fileKey.replace(/\s+/g, "_"),
                Body: compressedImage,
                ACL: "public-read",
            };
            const response = await s3.upload(params).promise();
            return new dataResponse_dto_1.DataResponseDto(response.Location, "Image uploaded successfully");
        }
        catch (error) {
            console.error(`Error uploading to S3: ${error.message}`);
            return new dataResponse_dto_1.DataResponseDto(null, `Error uploading to S3: ${error.message}`);
        }
    },
    deleteImage: async (queryParams) => {
        try {
            const { fileKey } = queryParams;
            if (!fileKey) {
                return new dataResponse_dto_1.DataResponseDto(null, "File key not found");
            }
            const url = new URL(fileKey);
            const path = url.pathname;
            const parts = path.split("/");
            const key = `${parts[1]}/${parts[2]}`;
            const params = {
                Bucket: process.env.S3_BUCKET,
                Key: key,
            };
            let response = await s3.deleteObject(params).promise();
            if (response === null || response === void 0 ? void 0 : response.DeleteMarker) {
                return new dataResponse_dto_1.DataResponseDto({}, "Image deleted successfully");
            }
            return new dataResponse_dto_1.DataResponseDto(null, "Failed to delete image");
        }
        catch (error) {
            console.error(`Error deleting from S3: ${error.message}`);
            return new dataResponse_dto_1.DataResponseDto(null, `Error deleting from S3: ${error.message}`);
        }
    },
    uploadImageorPdf: async (file, isCompress) => {
        try {
            if (!file.files.path) {
                return new dataResponse_dto_1.DataResponseDto(null, "Failed to upload file");
            }
            const fileType = file.files.type;
            let compressedFile;
            if (fileType.startsWith("image/")) {
                const imageData = await fs.promises.readFile(file.files.path);
                if (isCompress) {
                    const image = await Jimp.read(imageData);
                    const compressedImage = await image
                        .resize(Jimp.AUTO, 500)
                        .quality(60)
                        .getBufferAsync(Jimp.MIME_PNG);
                    compressedFile = compressedImage;
                }
                else {
                    compressedFile = imageData;
                }
            }
            else {
                compressedFile = await fs.promises.readFile(file.files.path);
            }
            const folderName = fileType.startsWith("image/") || fileType === "application/postscript"
                ? `${process.env.S3_FOLDER_NAME}/images`
                : `${process.env.S3_FOLDER_NAME}/pdfs`;
            const fileName = moment().format("YYYY-MM-DDHHmmss") +
                "-" +
                Math.random().toString(36).substring(2) + "-" +
                file.files.name;
            const fileKey = `${folderName}/${fileName}`;
            const params = {
                Bucket: process.env.S3_BUCKET,
                Key: fileKey.replace(/\s+/g, "_"),
                Body: compressedFile,
                ACL: "public-read",
                ContentType: fileType,
            };
            const response = await s3.upload(params).promise();
            return new dataResponse_dto_1.DataResponseDto(response.Location, "File uploaded successfully");
        }
        catch (error) {
            console.error(`Error uploading to S3: ${error.message}`);
            return new dataResponse_dto_1.DataResponseDto(null, `Error uploading to S3: ${error.message}`);
        }
    },
    uploadImageToAzure: async (imageFile) => {
        try {
            if (!imageFile.files || !imageFile.files.path) {
                return new dataResponse_dto_1.DataResponseDto(null, "Failed to upload image");
            }
            const imageData = await fs.promises.readFile(imageFile.files.path);
            const image = await Jimp.read(imageData);
            const compressedImage = await image
                .resize(Jimp.AUTO, 500)
                .quality(60)
                .getBufferAsync(Jimp.MIME_PNG);
            const title = imageFile.files.name.split(".")[0];
            const fileName = moment().format("YYYY-MM-DDHHmmss") + "-" + title + ".png";
            const blobName = `img/${fileName}`.replace(/\s+/g, "_");
            const account = process.env.AZURE_STORAGE_ACCOUNT;
            const accountKey = process.env.AZURE_STORAGE_ACCOUNT_KEY;
            const containerName = process.env.AZURE_STORAGE_CONTAINER;
            const azureUrl = process.env.AZURE_STORAGE_URL;
            const sharedKeyCredential = new StorageSharedKeyCredential(account, accountKey);
            const blobServiceClient = new BlobServiceClient(`${azureUrl}`, sharedKeyCredential);
            const containerClient = blobServiceClient.getContainerClient(containerName);
            const blockBlobClient = containerClient.getBlockBlobClient(blobName);
            await blockBlobClient.uploadData(compressedImage, {
                blobHTTPHeaders: {
                    blobContentType: "image/png",
                },
            });
            const fileUrl = `${azureUrl}/${containerName}/${blobName}`;
            return new dataResponse_dto_1.DataResponseDto(fileUrl, "Image uploaded to Azure successfully");
        }
        catch (error) {
            console.error(`Error uploading to Azure: ${error.message}`);
            return new dataResponse_dto_1.DataResponseDto(null, `Error uploading to Azure: ${error.message}`);
        }
    },
};
