"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
exports.default = strapi_1.factories.createCoreService("api::meeting-category.meeting-category", ({ strapi }) => ({
    async findOneCategory(id, params) {
        try {
            const { search } = params;
            const data = await strapi.db.query("api::meeting-category.meeting-category").findOne({
                where: {
                    id, is_deleted: false
                },
                populate: {
                    meeting_portal: true,
                    meeting_rooms: {
                        where: {
                            is_deleted: true,
                            ...(search && search.trim() && {
                                $or: [
                                    { name: { $containsi: search.trim() } },
                                    { location: { $containsi: search.trim() } },
                                ]
                            })
                        },
                        orderBy: { date: "asc" }
                    }
                }
            });
            if (!data) {
                return new dataResponse_dto_1.DataResponseDto(null, "");
            }
            return new dataResponse_dto_1.DataResponseDto(data, "meeting category data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async createCategoryWithRooms(body) {
        try {
            const knex = strapi.db.connection;
            const result = await strapi.db.transaction(async ({ trx }) => {
                if (!(body === null || body === void 0 ? void 0 : body.categoryTitle) || !(body === null || body === void 0 ? void 0 : body.userId) || !(body === null || body === void 0 ? void 0 : body.portalId)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "Provide all required fields");
                }
                const meetingCategory = await strapi.entityService.create("api::meeting-category.meeting-category", {
                    data: {
                        title: body === null || body === void 0 ? void 0 : body.categoryTitle,
                        notes: body === null || body === void 0 ? void 0 : body.categoryNotes,
                        portalId: body === null || body === void 0 ? void 0 : body.portalId,
                        venue: body === null || body === void 0 ? void 0 : body.location,
                        creator: body === null || body === void 0 ? void 0 : body.userId,
                        is_deleted: false,
                        meeting_portal: body === null || body === void 0 ? void 0 : body.meetingPortalId,
                        publishedAt: new Date()
                    },
                    transacting: trx
                });
                if ((body === null || body === void 0 ? void 0 : body.rooms) && body.rooms.length > 0) {
                    const meetingData = body.rooms.flatMap((room) => room.slots.map((slot) => ({
                        name: room.name,
                        date: new Date(room.date),
                        slot_time: slot.slotTime,
                        location: room.location,
                        capacity: room.capacity,
                        remaining_capacity: room.capacity,
                        ticket_holder_access: room.ticketHolderAccess,
                        status: "open",
                        is_deleted: false,
                        // creator: body.userId,
                        // meeting_category: meetingCategory?.id,
                        time_short: slot.timeShort,
                        published_at: new Date(),
                    })));
                    // meeting room
                    const createdMeetings = await knex("meeting_rooms").insert(meetingData).transacting(trx).returning('id');
                    const roomCreatorData = [];
                    const roomCategoryData = createdMeetings.map((item, index) => {
                        roomCreatorData.push({
                            meeting_room_id: item === null || item === void 0 ? void 0 : item.id,
                            user_id: body.userId,
                            meeting_room_order: Number(index + 1)
                        });
                        return {
                            meeting_room_id: item === null || item === void 0 ? void 0 : item.id,
                            meeting_category_id: meetingCategory === null || meetingCategory === void 0 ? void 0 : meetingCategory.id,
                            meeting_room_order: Number(index + 1)
                        };
                    });
                    // meeting_rooms_meeting_category_links
                    await knex("meeting_rooms_meeting_category_links").insert(roomCategoryData).transacting(trx);
                    // meeting_rooms_creator_links
                    await knex("meeting_rooms_creator_links").insert(roomCreatorData).transacting(trx);
                }
                return new dataResponse_dto_1.DataResponseDto(meetingCategory, "Meeting rooms created successfully");
            });
            return result;
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async updateCategoryWithRooms(id, body) {
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                var _a;
                const existingCategory = await strapi
                    .query("api::meeting-category.meeting-category")
                    .findOne({
                    where: { id, is_deleted: false }
                });
                if (!existingCategory) {
                    return new dataResponse_dto_1.DataResponseDto(null, "No meeting category found");
                }
                const meetingCategory = await strapi.entityService.update("api::meeting-category.meeting-category", id, {
                    data: {
                        title: body === null || body === void 0 ? void 0 : body.categoryTitle,
                        notes: body === null || body === void 0 ? void 0 : body.categoryNotes,
                        portalId: body === null || body === void 0 ? void 0 : body.portalId,
                        venue: body === null || body === void 0 ? void 0 : body.location,
                        modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                        meeting_portal: body === null || body === void 0 ? void 0 : body.meetingPortalId,
                    },
                    transacting: trx
                });
                if ((body === null || body === void 0 ? void 0 : body.rooms) && ((_a = body === null || body === void 0 ? void 0 : body.rooms) === null || _a === void 0 ? void 0 : _a.length) > 0) {
                    for (const room of body === null || body === void 0 ? void 0 : body.rooms) {
                        for (const slot of room === null || room === void 0 ? void 0 : room.slots) {
                            let remainingCapacity = 0;
                            if (Number(slot.capacity) === 0 || Number(slot.capacity)) {
                                let capacity_count = slot.capacity === 0 ? 0 : slot.capacity;
                                const existingSlot = await strapi
                                    .query("api::meeting-room.meeting-room")
                                    .findOne({
                                    where: { id: slot.id, is_deleted: false, status: "open" }
                                });
                                if (!existingSlot) {
                                    return new dataResponse_dto_1.DataResponseDto(null, `There is no slot for id ${slot.id}`);
                                }
                                remainingCapacity =
                                    Number(existingSlot === null || existingSlot === void 0 ? void 0 : existingSlot.capacity) -
                                        Number(existingSlot === null || existingSlot === void 0 ? void 0 : existingSlot.remainingCapacity);
                                if (Number(remainingCapacity) > Number(capacity_count)) {
                                    return new dataResponse_dto_1.DataResponseDto(null, "capacity cannot be less than remaining capacity");
                                }
                                if (Number(capacity_count) != Number(remainingCapacity)) {
                                    remainingCapacity =
                                        Number(capacity_count) - Number(remainingCapacity);
                                }
                                else {
                                    remainingCapacity = Number(capacity_count);
                                }
                            }
                            if (slot.id) {
                                await strapi.entityService.update("api::meeting-room.meeting-room", slot.id, {
                                    data: {
                                        name: room === null || room === void 0 ? void 0 : room.name,
                                        date: new Date(room === null || room === void 0 ? void 0 : room.date),
                                        slotTime: slot === null || slot === void 0 ? void 0 : slot.slotTime,
                                        location: room === null || room === void 0 ? void 0 : room.location,
                                        capacity: room === null || room === void 0 ? void 0 : room.capacity,
                                        remainingCapacity: Number(remainingCapacity),
                                        ticketHolderAccess: room === null || room === void 0 ? void 0 : room.ticketHolderAccess,
                                        status: slot === null || slot === void 0 ? void 0 : slot.status,
                                        modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                                        timeShort: slot === null || slot === void 0 ? void 0 : slot.timeShort
                                    },
                                    transacting: trx
                                });
                            }
                            else {
                                await strapi.entityService.create("api::meeting-room.meeting-room", {
                                    data: {
                                        name: room === null || room === void 0 ? void 0 : room.name,
                                        date: new Date(room === null || room === void 0 ? void 0 : room.date),
                                        slotTime: slot === null || slot === void 0 ? void 0 : slot.slotTime,
                                        location: room === null || room === void 0 ? void 0 : room.location,
                                        capacity: room === null || room === void 0 ? void 0 : room.capacity,
                                        remainingCapacity: room === null || room === void 0 ? void 0 : room.capacity,
                                        ticketHolderAccess: room === null || room === void 0 ? void 0 : room.ticketHolderAccess,
                                        status: "open",
                                        creator: body === null || body === void 0 ? void 0 : body.userId,
                                        meeting_category: meetingCategory === null || meetingCategory === void 0 ? void 0 : meetingCategory.id,
                                        timeShort: slot === null || slot === void 0 ? void 0 : slot.timeShort,
                                        publishedAt: new Date()
                                    },
                                    transacting: trx
                                });
                            }
                        }
                    }
                }
                return new dataResponse_dto_1.DataResponseDto(meetingCategory, "Meeting rooms updated successfully");
            });
            return result;
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findExportData() {
        try {
            const data = await strapi.db.query("api::meeting-category.meeting-category").findMany({
                where: { is_deleted: false },
                orderBy: { id: "desc" },
            });
            return new dataResponse_dto_1.DataResponseDto(data, "meeting-category data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllCategories(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                is_deleted: false,
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search &&
                    search.trim() && {
                    $or: [
                        { title: { $containsi: search } },
                        { location: { $containsi: search } }
                    ],
                }),
            };
            const data = await strapi.db.query("api::meeting-category.meeting-category").findMany({
                where,
                offset,
                limit: limit,
                orderBy: { id: "desc" },
                populate: {
                    creator: true,
                    modifiedBy: true
                }
            });
            if (data) {
                const totalDataCount = await strapi
                    .query("api::meeting-category.meeting-category")
                    .count({ where });
                const pageCount = Math.ceil(totalDataCount / limit);
                return new dataResponse_dto_1.DataResponseDto({
                    data,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount,
                        total: Number(totalDataCount),
                    },
                }, "meeting-category data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log("error", error);
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db
                .query("api::meeting-category.meeting-category")
                .update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            const deletedRooms = await strapi.db
                .query("api::meeting-room.meeting-room")
                .updateMany({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "meeting-category deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    }
}));
