"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const constants_1 = require("../../../shared/helpers/constants");
const admin = require('firebase-admin');
exports.default = strapi_1.factories.createCoreService("api::notification.notification", ({ strapi }) => ({
    // initialize firebase app
    initNotifications() {
        admin.initializeApp({
            credential: admin.credential.cert({
                "type": process.env.FIREBASE_ACCOUNT_TYPE,
                "project_id": process.env.FIREBASE_PROJECT_ID,
                "private_key_id": process.env.FIREBASE_PRIVATE_KEY_ID,
                "private_key": process.env.FIREBASE_PRIVATE_KEY.replace(/\\n/g, '\n')
                    .replace(/"/g, '')
                    .replace(/^"(.*)"$/, '$1'),
                "client_email": process.env.FIREBASE_CLIENT_EMAIL,
                "client_id": process.env.FIREBASE_CLIENT_ID,
                "auth_uri": process.env.FIREBASE_AUTH_URI,
                "token_uri": process.env.FIREBASE_TOKEN_URI,
                "auth_provider_x509_cert_url": process.env.FIREBASE_AUTH_PROVIDER_CERT_URL,
                "client_x509_cert_url": process.env.FIREBASE_CLIENT_CERT_URL,
                "universe_domain": process.env.FIREBASE_UNIVERSE_DOMAIN
            })
        });
    },
    async findAllNotifications(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                is_deleted: false,
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search &&
                    search.trim() && {
                    $or: [
                        { title: { $containsi: search } },
                    ],
                }),
            };
            const data = await strapi.db.query("api::notification.notification").findMany({
                where,
                offset,
                limit: limit,
                orderBy: { id: "desc" },
                populate: {
                    creator: {
                        select: ["firstName", "lastName"]
                    },
                    modifiedBy: {
                        select: ["firstName", "lastName"]
                    }
                }
            });
            if (data) {
                const totalDataCount = await strapi
                    .query("api::notification.notification")
                    .count({ where });
                const pageCount = Math.ceil(totalDataCount / limit);
                return new dataResponse_dto_1.DataResponseDto({
                    data,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount,
                        total: Number(totalDataCount),
                    },
                }, "notification data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log("error", error);
        }
    },
    async createNotification(body) {
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                const knex = strapi.db.connection;
                const notification = await strapi.db.query("api::notification.notification").create({
                    data: {
                        title: body === null || body === void 0 ? void 0 : body.title,
                        description: body === null || body === void 0 ? void 0 : body.description,
                        logo: body === null || body === void 0 ? void 0 : body.logo,
                        isScheduled: body === null || body === void 0 ? void 0 : body.isScheduled,
                        date: body === null || body === void 0 ? void 0 : body.date,
                        isSent: !(body === null || body === void 0 ? void 0 : body.isScheduled) ? true : body === null || body === void 0 ? void 0 : body.isSent,
                        creator: body === null || body === void 0 ? void 0 : body.userId,
                        is_deleted: false,
                        publishedAt: new Date()
                    }
                });
                if (!notification.isScheduled) {
                    // create read entries
                    const userDetails = await strapi.db.query("api::ticket-master.ticket-master").findMany({
                        where: {
                            is_deleted: false,
                            status: "issued",
                        },
                        select: ["fcmToken", "id"]
                    });
                    if (userDetails.length) {
                        const readData = userDetails.map((item) => ({
                            is_deleted: false,
                            is_read: false,
                            published_at: new Date(),
                            created_at: new Date()
                        }));
                        const notificationReads = await knex("notification_reads").insert(readData).transacting(trx).returning("id");
                        const notification_reads_notification = [];
                        const notification_reads_ticket_id_data = userDetails.map((user, index) => {
                            const notificationReadId = notificationReads[index].id;
                            notification_reads_notification.push({
                                notification_read_id: notificationReadId,
                                notification_id: notification.id,
                                notification_read_order: index + 1,
                            });
                            return {
                                ticket_master_id: user.id,
                                notification_read_id: notificationReadId,
                                notification_read_order: index + 1,
                            };
                        });
                        // 4. create entries to notification_reads_ticket_id_links table
                        await knex("notification_reads_ticket_id_links").insert(notification_reads_ticket_id_data).transacting(trx);
                        // 5. create entries to notification_reads_notification_id_links table
                        await knex("notification_reads_notification_id_links").insert(notification_reads_notification).transacting(trx);
                    }
                    const tokens = [...new Set(userDetails === null || userDetails === void 0 ? void 0 : userDetails.map(item => item === null || item === void 0 ? void 0 : item.fcmToken).filter(Boolean).flat())];
                    const batchSize = 490;
                    const batches = [];
                    for (let i = 0; i < tokens.length; i += batchSize) {
                        batches.push(tokens.slice(i, i + batchSize));
                    }
                    const message = {
                        notification: {
                            title: body === null || body === void 0 ? void 0 : body.title,
                            body: body === null || body === void 0 ? void 0 : body.description,
                            imageUrl: constants_1.IMAGES.PUSH_NOTIFICATION_IMAGE
                        }
                    };
                    for (const batchTokens of batches) {
                        const batchMessage = {
                            ...message,
                            tokens: batchTokens
                        };
                        await admin.messaging().sendEachForMulticast(batchMessage);
                    }
                }
                return new dataResponse_dto_1.DataResponseDto(notification, "Notification created successfully");
            });
            return result;
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async updateNotification(id, body) {
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                const knex = strapi.db.connection;
                // fetch the notification data with the given id
                const notification = await strapi.db.query("api::notification.notification").findOne({
                    where: {
                        id,
                        is_deleted: false,
                    }
                });
                if (!notification) {
                    return new dataResponse_dto_1.DataResponseDto(null, "No notification found for this id");
                }
                // update 
                const data = await strapi.db.query("api::notification.notification").update({
                    where: { id, is_deleted: false },
                    data: {
                        title: body === null || body === void 0 ? void 0 : body.title,
                        description: body === null || body === void 0 ? void 0 : body.description,
                        isScheduled: body === null || body === void 0 ? void 0 : body.isScheduled,
                        logo: body === null || body === void 0 ? void 0 : body.logo,
                        date: body === null || body === void 0 ? void 0 : body.date,
                        modifiedBy: body === null || body === void 0 ? void 0 : body.userId
                    },
                });
                if (!data.isScheduled && !data.isSent) {
                    // create read entries
                    const userDetails = await strapi.db.query("api::ticket-master.ticket-master").findMany({
                        where: {
                            is_deleted: false,
                            status: "issued",
                            fcmToken: { $notNull: true }
                        },
                        select: ["fcmToken", "id"]
                    });
                    if (userDetails.length) {
                        const readData = userDetails.map((item) => ({
                            is_deleted: false,
                            is_read: false,
                            published_at: new Date(),
                            created_at: new Date()
                        }));
                        const notificationReads = await knex("notification_reads").insert(readData).transacting(trx).returning("id");
                        const notification_reads_notification = [];
                        const notification_reads_ticket_id_data = userDetails.map((user, index) => {
                            const notificationReadId = notificationReads[index].id;
                            notification_reads_notification.push({
                                notification_read_id: notificationReadId,
                                notification_id: notification.id,
                                notification_read_order: index + 1,
                            });
                            return {
                                ticket_master_id: user.id,
                                notification_read_id: notificationReadId,
                                notification_read_order: index + 1,
                            };
                        });
                        // 4. create entries to notification_reads_ticket_id_links table
                        await knex("notification_reads_ticket_id_links").insert(notification_reads_ticket_id_data).transacting(trx);
                        // 5. create entries to notification_reads_notification_id_links table
                        await knex("notification_reads_notification_id_links").insert(notification_reads_notification).transacting(trx);
                    }
                    const tokens = [...new Set(userDetails === null || userDetails === void 0 ? void 0 : userDetails.map(item => item === null || item === void 0 ? void 0 : item.fcmToken).filter(Boolean).flat())];
                    const batchSize = 490;
                    const batches = [];
                    for (let i = 0; i < tokens.length; i += batchSize) {
                        batches.push(tokens.slice(i, i + batchSize));
                    }
                    const message = {
                        notification: {
                            title: body === null || body === void 0 ? void 0 : body.title,
                            body: body === null || body === void 0 ? void 0 : body.description,
                            imageUrl: constants_1.IMAGES.PUSH_NOTIFICATION_IMAGE
                        }
                    };
                    for (const batchTokens of batches) {
                        const batchMessage = {
                            ...message,
                            tokens: batchTokens
                        };
                        await admin.messaging().sendEachForMulticast(batchMessage);
                    }
                }
                return new dataResponse_dto_1.DataResponseDto(notification, "Notification updated successfully");
            });
            return result;
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async softDelete(id) {
        try {
            // fetch the notification data with the given id
            const notification = await strapi.db.query("api::notification.notification").findOne({
                where: {
                    id,
                    is_deleted: false,
                }
            });
            if (!notification) {
                return new dataResponse_dto_1.DataResponseDto(null, "No notification found for this id");
            }
            // update the soft delete field to true
            const data = await strapi.db.query("api::notification.notification").update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            return new dataResponse_dto_1.DataResponseDto(data, "Notification deleted successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async testCreateNotification(body) {
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                const knex = strapi.db.connection;
                const notification = await strapi.db.query("api::notification.notification").create({
                    data: {
                        title: body === null || body === void 0 ? void 0 : body.title,
                        description: body === null || body === void 0 ? void 0 : body.description,
                        logo: body === null || body === void 0 ? void 0 : body.logo,
                        isScheduled: (body === null || body === void 0 ? void 0 : body.isScheduled) || false,
                        date: (body === null || body === void 0 ? void 0 : body.date) || new Date(),
                        isSent: !(body === null || body === void 0 ? void 0 : body.isScheduled) ? true : body === null || body === void 0 ? void 0 : body.isSent,
                        creator: body === null || body === void 0 ? void 0 : body.userId,
                        is_deleted: false,
                        publishedAt: new Date()
                    }
                });
                let firebaseResponse;
                if (!notification.isScheduled) {
                    // create read entries
                    const ticketNumArray = ["BEFA0A5", "BA52203", "B35A090"];
                    if (body === null || body === void 0 ? void 0 : body.cpi) {
                        ticketNumArray.push("RA21706", "R3502A0");
                    }
                    const userDetails = await strapi.db.query("api::ticket-master.ticket-master").findMany({
                        where: {
                            is_deleted: false,
                            status: "issued",
                            ticketNumber: {
                                $in: ticketNumArray
                            }
                        },
                        select: ["fcmToken", "id"]
                    });
                    if (userDetails.length) {
                        const readData = userDetails.map((item) => ({
                            is_deleted: false,
                            is_read: false,
                            published_at: new Date(),
                            created_at: new Date()
                        }));
                        const notificationReads = await knex("notification_reads").insert(readData).transacting(trx).returning("id");
                        const notification_reads_notification = [];
                        const notification_reads_ticket_id_data = userDetails.map((user, index) => {
                            const notificationReadId = notificationReads[index].id;
                            notification_reads_notification.push({
                                notification_read_id: notificationReadId,
                                notification_id: notification.id,
                                notification_read_order: index + 1,
                            });
                            return {
                                ticket_master_id: user.id,
                                notification_read_id: notificationReadId,
                                notification_read_order: index + 1,
                            };
                        });
                        // 4. create entries to notification_reads_ticket_id_links table
                        await knex("notification_reads_ticket_id_links").insert(notification_reads_ticket_id_data).transacting(trx);
                        // 5. create entries to notification_reads_notification_id_links table
                        await knex("notification_reads_notification_id_links").insert(notification_reads_notification).transacting(trx);
                    }
                    const tokens = [...new Set(userDetails === null || userDetails === void 0 ? void 0 : userDetails.map(item => item === null || item === void 0 ? void 0 : item.fcmToken).filter(Boolean).flat())];
                    const message = {
                        notification: {
                            title: body === null || body === void 0 ? void 0 : body.title,
                            body: body === null || body === void 0 ? void 0 : body.description,
                            imageUrl: constants_1.IMAGES.PUSH_NOTIFICATION_IMAGE
                        },
                        android: {
                            notification: {
                                sound: 'default'
                            }
                        },
                        tokens: tokens
                    };
                    firebaseResponse = await admin.messaging().sendEachForMulticast(message);
                }
                return new dataResponse_dto_1.DataResponseDto({ notification, firebaseResponse: { successCount: (firebaseResponse === null || firebaseResponse === void 0 ? void 0 : firebaseResponse.successCount) || 0, failureCount: (firebaseResponse === null || firebaseResponse === void 0 ? void 0 : firebaseResponse.failureCount) || 0 } }, "Notification created successfully");
            });
            return result;
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
}));
