"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
const axios = require("axios");
exports.default = strapi_1.factories.createCoreService("api::order.order", ({ strapi }) => ({
    async findExportData(params) {
        try {
            const { filters } = params;
            const data = await strapi.db.query("api::order.order").findMany({
                where: {
                    is_deleted: false,
                    ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                },
                orderBy: { id: "desc" },
            });
            return new dataResponse_dto_1.DataResponseDto(data, "Order data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllOrders(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const { ticket_id, dayShort, ...restFilters } = filters;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const queryOptions = {
                where: {
                    is_deleted: false,
                    ...(restFilters && Object.keys(restFilters).length > 0 && { ...restFilters }),
                    ...(search &&
                        search.trim().length && {
                        $or: [
                            { orderId: { $containsi: search } },
                            { firstName: { $containsi: search } },
                            { lastName: { $containsi: search } },
                            //  expect only two words
                            {
                                $and: [
                                    { firstName: { $containsi: search.split(' ')[0] } },
                                    { lastName: { $containsi: search.split(' ')[1] } },
                                ],
                            },
                            { email: { $containsi: search } },
                        ],
                    }),
                    ticket_masters: {
                        ...(ticket_id && { ticket_id: { $eq: Number(ticket_id) } }),
                        ...(dayShort && { dayShort: { $eq: dayShort } }),
                    },
                },
                populate: {
                    creator: {
                        select: ["username", "firstName", "lastName"]
                    },
                    ticket_masters: {
                        where: {
                            ...(ticket_id && { ticket_id: { $eq: Number(ticket_id) } }),
                            ...(dayShort && { dayShort: { $eq: dayShort } }),
                        },
                    },
                },
                offset,
                limit: limit,
                orderBy: { createdAt: "desc" },
            };
            const data = await strapi.db.query("api::order.order").findMany(queryOptions);
            if (data) {
                const totalDataCount = await strapi
                    .query("api::order.order")
                    .count(queryOptions);
                const pageCount = Math.ceil(totalDataCount / limit);
                return new dataResponse_dto_1.DataResponseDto({
                    data,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount,
                        total: totalDataCount,
                    },
                }, "order data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log("error", error);
        }
    },
    async findOrderDetails(id) {
        try {
            const data = await strapi.db.query("api::order.order").findOne({
                where: { id, is_deleted: false },
                populate: {
                    discountCode: true,
                },
            });
            if (!data) {
                return new dataResponse_dto_1.DataResponseDto(null, "Order not found");
            }
            const tickets = await strapi.db
                .query("api::ticket-master.ticket-master")
                .findMany({
                where: { orderId: id, is_deleted: false,
                    status: {
                        $notIn: [
                            "refunded", "cancelled", "deleted",
                            "draft",
                        ],
                    },
                },
                populate: {
                    ticket_id: {
                        select: ["title"],
                    },
                },
            });
            const dataWithTickets = { ...data, tickets };
            return new dataResponse_dto_1.DataResponseDto(dataWithTickets, "Order details fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async checkTicketData(email, orderNumber) {
        try {
            if (!orderNumber) {
                const data = await strapi.db
                    .query("api::ticket-master.ticket-master")
                    .findMany({
                    where: { email: email, is_deleted: false },
                });
                if (data && data.length > 0) {
                    return {
                        data,
                        status: false,
                        message: "Ticket already assigned to this email",
                    };
                }
                return new dataResponse_dto_1.DataResponseDto([], true, "There is no ticket assigned to this email");
            }
            const ticket = await strapi.db
                .query("api::ticket-master.ticket-master")
                .findMany({
                where: {
                    orderNumber: orderNumber,
                    email: email,
                    status: {
                        $in: ["assigned", "reassigned"]
                    },
                    is_deleted: false,
                },
            });
            if (ticket && ticket.length > 0) {
                return {
                    data: ticket,
                    status: false,
                    message: "Ticket already assigned to this email",
                };
            }
            return new dataResponse_dto_1.DataResponseDto([], true, "There is no ticket assigned to this email");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db.query("api::order.order").update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "order deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async updateOrderStatus(OrderNumber, status, userAccess, userId) {
        var _a, _b;
        try {
            let refundRole = 28;
            let cancelRole = 29;
            let deleteRole = 30;
            const hasDeleteAccess = userAccess === null || userAccess === void 0 ? void 0 : userAccess.includes(deleteRole);
            const hasRefundAccess = userAccess === null || userAccess === void 0 ? void 0 : userAccess.includes(refundRole);
            const hasCancelAccess = userAccess === null || userAccess === void 0 ? void 0 : userAccess.includes(cancelRole);
            if (!hasDeleteAccess && status === "Deleted") {
                return new dataResponse_dto_1.DataResponseDto(null, "You don't have delete access");
            }
            if (!hasRefundAccess && status === "Refunded") {
                return new dataResponse_dto_1.DataResponseDto(null, "You don't have refund access");
            }
            if (!hasCancelAccess && status === "Cancelled") {
                return new dataResponse_dto_1.DataResponseDto(null, "You don't have cancel access");
            }
            const orderData = await strapi.db.query("api::order.order").findOne({
                where: { orderId: OrderNumber, is_deleted: false },
            });
            if (!orderData) {
                return new dataResponse_dto_1.DataResponseDto(null, "No data found");
            }
            const ticketsData = await strapi.db.query("api::ticket-master.ticket-master").findMany({
                where: { orderNumber: OrderNumber, is_deleted: false },
                populate: {
                    ticket_id: true
                }
            });
            let orderObj = {
                status: status,
                modifiedBy: userId,
            };
            if (status === "Deleted") {
                orderObj.is_deleted = true;
            }
            const data = await strapi.db.query("api::order.order").update({
                where: { orderId: OrderNumber, is_deleted: false },
                data: orderObj,
            });
            let ticketObj = {
                status: status === "Refunded"
                    ? "refunded"
                    : status === "Deleted"
                        ? "deleted"
                        : "cancelled",
                modifiedBy: userId,
                token: "",
                allowChat: false
            };
            if (status === "Deleted") {
                ticketObj.is_deleted = true;
            }
            // update tickets status
            const updatedTickets = await strapi.db
                .query("api::ticket-master.ticket-master")
                .updateMany({
                where: { orderNumber: OrderNumber },
                data: ticketObj,
            });
            // // update tickets in evento
            if (ticketsData.length) {
                for (const element of ticketsData) {
                    if ((element === null || element === void 0 ? void 0 : element.status) === "issued") {
                        let eventoObj = {
                            OrderNumber: "",
                            TicketId: element.ticketNumber,
                            Status: 0,
                        };
                        let url = process.env.EVENTO_UPDATE_STATUS;
                        await axios.post(url, eventoObj, {
                            headers: {
                                Authorization: process.env.EVENTO_VISITOR_TOKEN,
                                "Content-Type": "application/json",
                            },
                        });
                        if ((element === null || element === void 0 ? void 0 : element.status) === "issued" && element.ticket_id.id_short !== "general_admission") {
                            try {
                                const senbirdUrl = `https://api-${process.env.SENBIRD_APP_ID}.sendbird.com/v3/users/${element.ticketNumber}`;
                                const senbirdUser = await axios.get(senbirdUrl, {
                                    headers: {
                                        "Api-Token": process.env.SEND_BIRD_API_TOKEN,
                                        "Content-Type": "application/json",
                                    },
                                });
                                // Only attempt deletion if user exists
                                if ((senbirdUser === null || senbirdUser === void 0 ? void 0 : senbirdUser.data) && !((_a = senbirdUser === null || senbirdUser === void 0 ? void 0 : senbirdUser.data) === null || _a === void 0 ? void 0 : _a.error)) {
                                    await axios.delete(senbirdUrl, {
                                        headers: {
                                            "Api-Token": process.env.SEND_BIRD_API_TOKEN,
                                            "Content-Type": "application/json",
                                        },
                                    });
                                }
                            }
                            catch (error) {
                                // Ignore SendBird errors - user might not exist
                                console.log("SendBird operation failed:", error.message);
                            }
                        }
                    }
                }
            }
            ;
            // update order in evento
            let evnetoOrderStatus = false;
            if ((orderData === null || orderData === void 0 ? void 0 : orderData.status) === "Paid" && Number(orderData === null || orderData === void 0 ? void 0 : orderData.paidAmount) > 0) {
                let url = process.env.EVENTO_UPDATE_STATUS;
                let reqObj = {
                    OrderNumber: OrderNumber,
                    Status: status === "Refunded" ? -1 : 0,
                };
                const orderUpdate = await axios.post(url, reqObj, {
                    headers: {
                        Authorization: process.env.EVENTO_VISITOR_TOKEN,
                        "Content-Type": "application/json",
                    },
                });
                evnetoOrderStatus = (_b = orderUpdate === null || orderUpdate === void 0 ? void 0 : orderUpdate.data) === null || _b === void 0 ? void 0 : _b.Status;
            }
            return new dataResponse_dto_1.DataResponseDto({ order: data, evnetoOrderStatus, tickets: updatedTickets }, "order status updated successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
}));
