"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
exports.default = strapi_1.factories.createCoreService("api::payment-log.payment-log", ({ strapi }) => ({
    async createLog(body) {
        try {
            const existingOrderLog = await strapi
                .query("api::payment-log.payment-log")
                .findOne({
                where: { orderId: body.orderId, status: "pending" },
            });
            if (existingOrderLog)
                return new dataResponse_dto_1.DataResponseDto(null, "Log already exists for this order number");
            const data = await strapi.entityService.create("api::payment-log.payment-log", {
                data: {
                    orderId: body.orderId,
                    amount: body.amount,
                    status: "pending",
                    requestData: body.requestData,
                    publishedAt: new Date(),
                },
            });
            return new dataResponse_dto_1.DataResponseDto(data, "payment-log created successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async updateLogStatus(orderId) {
        try {
            const data = await strapi.query("api::payment-log.payment-log").update({
                where: { orderId: orderId },
                data: { status: "rejected" },
            });
            if (!data)
                return new dataResponse_dto_1.DataResponseDto(null, "data not found");
            return new dataResponse_dto_1.DataResponseDto(data, "data updated successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findExportData() {
        try {
            const data = await strapi.db.connection.raw(`
          SELECT DISTINCT ON (pl.request_data->>'Email') 
            pl.request_data->>'Email' AS email, 
            pl.request_data, 
            pl.id, 
            pl.amount, 
            pl.order_id
          FROM payment_logs AS pl
          WHERE pl.status = 'pending'
          AND pl.request_data->>'Email' IS NOT NULL
          ORDER BY pl.request_data->>'Email', pl.id DESC
        `);
            return new dataResponse_dto_1.DataResponseDto(data.rows, "payment-log data fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllLogs(params) {
        try {
            const { page, pageSize, search } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const knex = strapi.db.connection;
            const query = `
       WITH ranked_logs AS (
        SELECT 
          pl.request_data->>'Email' AS email,
          pl.request_data,
          pl.id,
          pl.amount,
          pl.order_id,
          pl.created_at,
          ROW_NUMBER() OVER (
            PARTITION BY pl.request_data->>'Email' 
            ORDER BY pl.created_at DESC
          ) AS row_num
        FROM payment_logs AS pl
        WHERE pl.status = 'pending'
          AND pl.request_data->>'Email' IS NOT NULL
          AND (
            pl.request_data->>'firstName' ILIKE ?
            OR pl.request_data->>'lastName' ILIKE ?
            OR pl.request_data->>'Email' ILIKE ?
            OR pl.order_id ILIKE ?
          )
      )
      SELECT 
        email,
        request_data,
        id,
        amount,
        order_id,
        created_at
      FROM ranked_logs
      WHERE row_num = 1
      ORDER BY created_at DESC
      LIMIT ? OFFSET ?;
      `;
            const data = await knex.raw(query, [
                `%${search}%`,
                `%${search}%`,
                `%${search}%`,
                `%${search}%`,
                limit,
                offset
            ]);
            if (data) {
                const query2 = `
        WITH ranked_logs AS (
        SELECT 
          pl.request_data->>'Email' AS email,
          pl.request_data,
          pl.id,
          pl.amount,
          pl.order_id,
          pl.created_at,
          ROW_NUMBER() OVER (
            PARTITION BY pl.request_data->>'Email' 
            ORDER BY pl.created_at DESC
          ) AS row_num
        FROM payment_logs AS pl
        WHERE pl.status = 'pending'
          AND pl.request_data->>'Email' IS NOT NULL
          AND (
            pl.request_data->>'firstName' ILIKE ?
            OR pl.request_data->>'lastName' ILIKE ?
            OR pl.request_data->>'Email' ILIKE ?
            OR pl.order_id ILIKE ?
          )
      )
      SELECT count(*)
      FROM ranked_logs
      WHERE row_num = 1
      `;
                const totalDataCountResult = await knex.raw(query2, [
                    `%${search}%`,
                    `%${search}%`,
                    `%${search}%`,
                    `%${search}%`
                ]);
                const totalDataCount = Number(totalDataCountResult.rows[0].count);
                const pageCount = Math.ceil(totalDataCount / limit);
                return new dataResponse_dto_1.DataResponseDto({
                    data: data.rows,
                    pagination: {
                        page: Number(pageNumber),
                        pageSize: Number(limit),
                        pageCount,
                        total: totalDataCount,
                    },
                }, "payment-log data fetched successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log("error", error);
        }
    },
    async syncSalesData(body) {
        var _a, _b;
        try {
            if (!(body === null || body === void 0 ? void 0 : body.orderIdArray) || (body === null || body === void 0 ? void 0 : body.orderIdArray.length) === 0) {
                return new dataResponse_dto_1.DataResponseDto("orderIdArray is empty or not provided");
            }
            const placeholders = body === null || body === void 0 ? void 0 : body.orderIdArray.map(() => '?').join(',');
            let query = `
          SELECT *,
          COUNT(*) OVER () AS total_count,
          SUM(pl.amount) OVER () AS total_sum
          FROM public.payment_logs pl
          WHERE pl.order_id IN (${placeholders})
            AND NOT EXISTS (
              SELECT 1
              FROM public.orders o
              WHERE o.order_id = pl.order_id
          )
          ORDER BY pl.id ASC;
        `;
            const knex = strapi.db.connection;
            const data = await knex.raw(query, body === null || body === void 0 ? void 0 : body.orderIdArray);
            return {
                dataCount: (_a = data === null || data === void 0 ? void 0 : data.rows) === null || _a === void 0 ? void 0 : _a.length,
                totalAmountMismatch: (_b = data === null || data === void 0 ? void 0 : data.rows[0]) === null || _b === void 0 ? void 0 : _b.total_sum,
                data: data === null || data === void 0 ? void 0 : data.rows,
                status: true,
                message: "Data fetched successfully"
            };
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
}));
