"use strict";
/**
 * restaurant service
 */
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
exports.default = strapi_1.factories.createCoreService("api::restaurant.restaurant", ({ strapi }) => ({
    async findAllRestaurants(params) {
        try {
            const { page, pageSize, search, filters } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const where = {
                is_deleted: false,
                ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                ...(search &&
                    search.trim() && {
                    $or: [
                        { name: { $containsi: search } },
                    ],
                }),
            };
            const restaurants = await strapi.db.query("api::restaurant.restaurant").findMany({
                where,
                offset,
                limit: limit,
                orderBy: { id: "desc" },
                populate: {
                    location: {
                        select: ["id", "name"]
                    },
                    creator: {
                        select: ["username", "firstName", "lastName"]
                    },
                    modifiedBy: {
                        select: ["username", "firstName", "lastName"]
                    },
                    menu_items: {
                        populate: {
                            category_id: {
                                select: ["id", "name"]
                            }
                        }
                    }
                }
            });
            const totalDataCount = await strapi
                .query("api::restaurant.restaurant")
                .count({ where });
            const pageCount = Math.ceil(totalDataCount / limit);
            return {
                data: restaurants,
                status: true,
                message: "Restaurants fetched successfully",
                pagination: {
                    page: Number(pageNumber),
                    pageSize: Number(limit),
                    pageCount,
                    total: Number(totalDataCount)
                }
            };
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong on our end; please contact the system administrator");
        }
    },
    async findAllByLocation(params) {
        try {
            const { page, pageSize, search } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            let data = await strapi.db.connection('venues as v')
                .select('v.id AS venue_id', 'v.image AS venue_image', 'v.name AS venue_name', 'v.banner_logo AS venue_banner_logo', 'r.id AS restaurant_id', 'r.name AS restaurant_name', 'r.logo AS restaurant_logo', 'r.description AS restaurant_description', 'r.perma_link as restaurant_perma_link', 'vrl.*')
                .join('restaurants_location_links as vrl', 'v.id', 'vrl.venue_id')
                .join('restaurants as r', 'vrl.restaurant_id', 'r.id')
                .where({
                'v.is_deleted': false,
                'r.is_deleted': false
            })
                .modify(function (queryBuilder) {
                if (search) {
                    queryBuilder.whereRaw('(v.name ILIKE ? OR r.name ILIKE ?)', [`%${search}%`, `%${search}%`]);
                }
            })
                .limit(limit)
                .offset(offset)
                .orderBy('v.id', 'desc');
            ;
            // fetch the count of all data 
            const dataCount = await strapi.db.connection('venues as v')
                .count('v.id as count')
                .join('restaurants_location_links as vrl', 'v.id', 'vrl.venue_id')
                .join('restaurants as r', 'vrl.restaurant_id', 'r.id')
                .where({
                'v.is_deleted': false,
                'r.is_deleted': false
            })
                .modify(function (queryBuilder) {
                if (search) {
                    queryBuilder.whereRaw('(v.name ILIKE ? OR r.name ILIKE ?)', [`%${search}%`, `%${search}%`]);
                }
            });
            // total count of data
            const totalDataCount = Number(dataCount[0].count);
            // calculate the total page count
            const pageCount = Math.ceil(totalDataCount / limit);
            // group the dat by location id
            const groupedData = data.reduce((acc, item) => {
                const venueIndex = acc.findIndex(venue => venue.venue_id === item.venue_id);
                if (venueIndex !== -1) {
                    acc[venueIndex].restaurants.push({
                        restaurant_id: item.restaurant_id,
                        restaurant_name: item.restaurant_name,
                        restaurant_logo: item.restaurant_logo,
                        restaurant_description: item.restaurant_description,
                        restaurant_order: item.restaurant_order,
                        restaurant_perma_link: item.restaurant_perma_link
                    });
                }
                else {
                    acc.push({
                        venue_id: item.venue_id,
                        venue_name: item.venue_name,
                        venue_image: item.venue_image,
                        venue_banner_logo: item.venue_banner_logo,
                        restaurants: [{
                                restaurant_id: item.restaurant_id,
                                restaurant_name: item.restaurant_name,
                                restaurant_logo: item.restaurant_logo,
                                restaurant_description: item.restaurant_description,
                                restaurant_order: item.restaurant_order,
                                restaurant_perma_link: item.restaurant_perma_link
                            }]
                    });
                }
                return acc;
            }, []);
            return {
                data: groupedData,
                status: true,
                message: "restaurant data fetched successfully",
                pagination: {
                    page: Number(pageNumber),
                    pageSize: Number(limit),
                    pageCount,
                    total: totalDataCount,
                }
            };
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong on our end; please contact the system administrator");
        }
    },
    async findAllItems(id, params) {
        try {
            const { page, pageSize, search } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            const restaurant = await strapi.db
                .query("api::restaurant.restaurant")
                .findOne({
                where: { id, is_deleted: false },
                select: ["name", "logo", "bannerImage", "image1"]
            });
            if (!restaurant) {
                return new dataResponse_dto_1.DataResponseDto(null, "No restaurant found");
            }
            const restaurantsItems = await strapi.db.connection('menu_categories as mc')
                .select('mc.id as category_id', 'mc.name as category_name', 'mc.value as category_value', 'mc.image as category_image', strapi.db.connection.raw(`
            JSON_AGG(
              DISTINCT JSONB_BUILD_OBJECT(
                'item_id', mi.id,
                'item_name', mi.name,
                'item_image', mi.image,
                'item_price', mi.price,
                'item_description', mi.description,
                'item_is_available', mi.is_available
              )
            ) AS items
          `))
                .join('menu_items_category_id_links as micil', 'micil.menu_category_id', 'mc.id')
                .join('menu_items as mi', 'micil.menu_item_id', 'mi.id')
                .join('menu_items_restaurant_links as mirl', 'mirl.menu_item_id', 'mi.id')
                .where({ 'mi.is_deleted': false, 'mirl.restaurant_id': id })
                .modify(queryBuilder => {
                if (search) {
                    queryBuilder.whereRaw('mc.name ILIKE ?', [`%${search}%`]);
                }
            })
                .groupBy('mc.id', 'mc.name')
                .limit(limit)
                .offset(offset);
            const dataCount = await strapi.db.connection('menu_categories as mc')
                .countDistinct("mc.id as count")
                .join('menu_items_category_id_links as micil', 'micil.menu_category_id', 'mc.id')
                .join('menu_items as mi', 'micil.menu_item_id', 'mi.id')
                .join('menu_items_restaurant_links as mirl', 'micil.menu_item_id', 'mi.id')
                .where({ 'mi.is_deleted': false, 'mirl.restaurant_id': id })
                .modify(queryBuilder => {
                if (search) {
                    queryBuilder.whereRaw('mc.name ILIKE ?', [`%${search}%`]);
                }
            })
                .first();
            const totalDataCount = Number(dataCount.count);
            const pageCount = Math.ceil(totalDataCount / limit);
            return {
                data: {
                    ...restaurant,
                    items: restaurantsItems
                },
                status: true,
                message: "Menu items fetched successfully",
                pagination: {
                    page: Number(pageNumber),
                    pageSize: limit,
                    pageCount,
                    total: totalDataCount,
                },
            };
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong on our end; please contact the system administrator");
        }
    },
    async findAllByLocationApp(params) {
        try {
            const { page, pageSize, search } = params;
            let limit = Number(pageSize) || 10;
            let pageNumber = Number(page) || 1;
            const offset = (Number(pageNumber) - 1) * Number(limit);
            let data = await strapi.db.connection('venues as v')
                .select('v.id AS venue_id', 'v.image AS venue_image', 'v.name AS venue_name', 'v.banner_logo AS venue_banner_logo', strapi.db.connection.raw(`
            JSON_AGG(
              DISTINCT JSONB_BUILD_OBJECT(
                'restaurant_id', r.id,
                'restaurant_name', r.name,
                'restaurant_logo', r.logo,
                'restaurant_description', r.description,
                'restaurant_perma_link', r.perma_link
              )
            ) AS restaurants
          `))
                .join('restaurants_location_links as vrl', 'v.id', 'vrl.venue_id')
                .join('restaurants as r', 'vrl.restaurant_id', 'r.id')
                .where({
                'v.is_deleted': false,
                'r.is_deleted': false
            })
                .modify(function (queryBuilder) {
                if (search) {
                    queryBuilder.whereRaw('(v.name ILIKE ? OR r.name ILIKE ?)', [`%${search}%`, `%${search}%`]);
                }
            })
                .groupBy('v.id', 'v.image', 'v.name', 'v.banner_logo')
                .limit(limit)
                .offset(offset)
                .orderBy('v.id', 'desc');
            ;
            // fetch the count of all data 
            const dataCount = await strapi.db.connection('venues as v')
                .count('v.id as count')
                .join('restaurants_location_links as vrl', 'v.id', 'vrl.venue_id')
                .join('restaurants as r', 'vrl.restaurant_id', 'r.id')
                .where({
                'v.is_deleted': false,
                'r.is_deleted': false
            })
                .modify(function (queryBuilder) {
                if (search) {
                    queryBuilder.whereRaw('(v.name ILIKE ? OR r.name ILIKE ?)', [`%${search}%`, `%${search}%`]);
                }
            });
            // total count of data
            const totalDataCount = Number(dataCount[0].count);
            // calculate the total page count
            const pageCount = Math.ceil(totalDataCount / limit);
            return {
                data,
                status: true,
                message: "Restaurants fetched successfully",
                pagination: {
                    page: Number(pageNumber),
                    pageSize: limit,
                    pageCount,
                    total: totalDataCount,
                },
            };
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong on our end; please contact the system administrator");
        }
    },
    async createRestaurantMenus(body) {
        var _a;
        try {
            if (!(body === null || body === void 0 ? void 0 : body.name) || !body.userId || !(body === null || body === void 0 ? void 0 : body.items)) {
                return new dataResponse_dto_1.DataResponseDto(null, "Provide all required fields");
            }
            const existingPermaLink = await strapi.db
                .query("api::restaurant.restaurant")
                .findMany({
                where: {
                    permaLink: (_a = body === null || body === void 0 ? void 0 : body.permaLink) === null || _a === void 0 ? void 0 : _a.trim(),
                    is_deleted: false
                },
            });
            if (existingPermaLink.length) {
                return new dataResponse_dto_1.DataResponseDto(null, "perma link already exists");
            }
            const result = await strapi.db.transaction(async ({ trx }) => {
                const restaurant = await strapi.entityService.create("api::restaurant.restaurant", {
                    data: {
                        name: body.name || "",
                        logo: body.logo || "",
                        bannerImage: (body === null || body === void 0 ? void 0 : body.bannerImage) || "",
                        description: body.description || "",
                        location: body.location,
                        is_deleted: false,
                        image1: (body === null || body === void 0 ? void 0 : body.image1) || "",
                        permaLink: (body === null || body === void 0 ? void 0 : body.permaLink) || "",
                        creator: body.userId,
                        publishedAt: new Date(),
                    },
                    transacting: trx,
                });
                // Create menu items if provided
                if (body.items && body.items.length > 0) {
                    const menuItemsPromises = body === null || body === void 0 ? void 0 : body.items.map(async (item) => {
                        return await strapi.entityService.create("api::menu-item.menu-item", {
                            data: {
                                name: (item === null || item === void 0 ? void 0 : item.name) || "",
                                image: (item === null || item === void 0 ? void 0 : item.image) || "",
                                price: Number(item === null || item === void 0 ? void 0 : item.price) || 0,
                                creator: body === null || body === void 0 ? void 0 : body.userId,
                                description: (item === null || item === void 0 ? void 0 : item.description) || "",
                                is_available: true,
                                is_deleted: false,
                                category_id: item.category,
                                restaurant: restaurant.id,
                                publishedAt: new Date(),
                            },
                            transacting: trx,
                        });
                    });
                    await Promise.all(menuItemsPromises);
                }
                return new dataResponse_dto_1.DataResponseDto(restaurant, "Menu created successfully");
            });
            return result;
        }
        catch (error) {
            console.error(error);
            console.log(JSON.stringify(error.details));
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong on our end; please contact the system administrator");
        }
    },
    async updateRestaurantMenus(id, body) {
        try {
            const result = await strapi.db.transaction(async ({ trx }) => {
                var _a;
                if (!id || !(body === null || body === void 0 ? void 0 : body.userId) || !(body === null || body === void 0 ? void 0 : body.items)) {
                    return new dataResponse_dto_1.DataResponseDto(null, "Provide all required fields");
                }
                const existingRestaurant = await strapi
                    .query("api::restaurant.restaurant")
                    .findOne({ where: { id }, populate: true });
                // Check if the restaurant exists
                if (!existingRestaurant) {
                    return new dataResponse_dto_1.DataResponseDto(null, "Restaurant not found");
                }
                const existingPermaLink = await strapi.db
                    .query("api::restaurant.restaurant")
                    .findMany({
                    where: {
                        permaLink: (_a = body === null || body === void 0 ? void 0 : body.permaLink) === null || _a === void 0 ? void 0 : _a.trim(),
                        is_deleted: false,
                        id: {
                            $ne: id
                        }
                    },
                });
                if (existingPermaLink.length) {
                    return new dataResponse_dto_1.DataResponseDto(null, "perma link already exists");
                }
                const updatedRestaurant = await strapi.entityService.update("api::restaurant.restaurant", id, {
                    data: {
                        name: (body === null || body === void 0 ? void 0 : body.name) || "",
                        logo: (body === null || body === void 0 ? void 0 : body.logo) || "",
                        bannerImage: (body === null || body === void 0 ? void 0 : body.bannerImage) || "",
                        description: (body === null || body === void 0 ? void 0 : body.description) || "",
                        location: body === null || body === void 0 ? void 0 : body.location,
                        image1: (body === null || body === void 0 ? void 0 : body.image1) || "",
                        permaLink: (body === null || body === void 0 ? void 0 : body.permaLink) || "",
                        modifiedBy: body === null || body === void 0 ? void 0 : body.userId,
                        publishedAt: new Date(),
                    },
                    transacting: trx,
                });
                // Update or create menu items based on the body
                if (body.items && body.items.length > 0) {
                    const menuItemsPromises = body.items.map(async (item) => {
                        if (item.id) {
                            // Update existing menu item
                            return await strapi.entityService.update("api::menu-item.menu-item", item.id, {
                                data: {
                                    name: (item === null || item === void 0 ? void 0 : item.name) || "",
                                    image: (item === null || item === void 0 ? void 0 : item.image) || "",
                                    price: (item === null || item === void 0 ? void 0 : item.price) || 0,
                                    creator: body === null || body === void 0 ? void 0 : body.userId,
                                    description: (item === null || item === void 0 ? void 0 : item.description) || "",
                                    is_available: item === null || item === void 0 ? void 0 : item.is_available,
                                    is_deleted: false,
                                    category_id: item === null || item === void 0 ? void 0 : item.category,
                                    publishedAt: new Date(),
                                },
                                transacting: trx,
                            });
                        }
                        else {
                            // Create new menu item if ID is not provided
                            return await strapi.entityService.create("api::menu-item.menu-item", {
                                data: {
                                    name: (item === null || item === void 0 ? void 0 : item.name) || "",
                                    image: (item === null || item === void 0 ? void 0 : item.image) || "",
                                    price: (item === null || item === void 0 ? void 0 : item.price) || 0,
                                    creator: body === null || body === void 0 ? void 0 : body.userId,
                                    description: (item === null || item === void 0 ? void 0 : item.description) || "",
                                    is_available: true,
                                    is_deleted: false,
                                    category_id: item === null || item === void 0 ? void 0 : item.category,
                                    restaurant: id,
                                    publishedAt: new Date(),
                                },
                                transacting: trx,
                            });
                        }
                    });
                }
                return new dataResponse_dto_1.DataResponseDto(updatedRestaurant, "Menu updated successfully");
            });
            return result;
        }
        catch (error) {
            console.error(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong on our end, please contact the system administrator");
        }
    },
    async softDelete(id) {
        try {
            const data = await strapi.db
                .query("api::restaurant.restaurant")
                .update({
                where: { id, is_deleted: false },
                data: {
                    is_deleted: true,
                },
            });
            if (data) {
                return new dataResponse_dto_1.DataResponseDto(data, "payment deleted successfully");
            }
            else {
                return new dataResponse_dto_1.DataResponseDto(null);
            }
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
}));
