"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const strapi_1 = require("@strapi/strapi");
const dataResponse_dto_1 = require("../../../shared/dto/dataResponse.dto");
exports.default = strapi_1.factories.createCoreService("api::ticket-benefit.ticket-benefit", ({ strapi }) => ({
    async findAllBenefitsByTicketNumber(ticketNumber, params) {
        var _a, _b;
        try {
            const { filters } = params;
            // fetch ticket data with type
            const ticketData = await strapi.db
                .query("api::ticket-master.ticket-master")
                .findOne({
                where: {
                    ticketNumber,
                    status: "issued",
                    is_deleted: false,
                },
                select: ["dayShort"],
                populate: {
                    ticket_id: {
                        select: ["id", "id_short"],
                    },
                },
            });
            if (!ticketData) {
                return new dataResponse_dto_1.DataResponseDto(null, "No ticket found for this ticket number");
            }
            let whereCase = {
                is_deleted: false,
                ticket_type: (_a = ticketData === null || ticketData === void 0 ? void 0 : ticketData.ticket_id) === null || _a === void 0 ? void 0 : _a.id,
                ...(filters && Object.keys(filters).length > 0 && { ...filters })
            };
            if (((_b = ticketData === null || ticketData === void 0 ? void 0 : ticketData.ticket_id) === null || _b === void 0 ? void 0 : _b.id_short) === "single_day_delegate") {
                whereCase["dayShort"] = ticketData === null || ticketData === void 0 ? void 0 : ticketData.dayShort;
            }
            const data = await strapi.db.query("api::ticket-benefit.ticket-benefit").findMany({
                where: whereCase,
                populate: {
                    benefits: {
                        where: {
                            is_deleted: false
                        },
                        select: ["name"],
                        populate: {
                            venue: {
                                select: ["name"]
                            }
                        }
                    },
                    ticket_type: true
                }
            });
            const groupedByDay = data.reduce((acc, item) => {
                const day = item.day;
                const existingDay = acc.find(entry => entry.day === day);
                if (existingDay) {
                    existingDay.benefits.push(...item.benefits);
                }
                else {
                    acc.push({ day: day, benefits: [...item.benefits] });
                }
                return acc;
            }, []);
            const result = groupedByDay.sort((a, b) => {
                const dayA = parseInt(a.day.match(/Day (\d+)/)[1], 10);
                const dayB = parseInt(b.day.match(/Day (\d+)/)[1], 10);
                return dayA - dayB;
            });
            return new dataResponse_dto_1.DataResponseDto(result, "Benefits fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async findAllBenefits(params) {
        try {
            const { filters } = params;
            const data = await strapi.db.query("api::ticket-benefit.ticket-benefit").findMany({
                where: {
                    ...(filters && Object.keys(filters).length > 0 && { ...filters }),
                    is_deleted: false
                },
                populate: {
                    benefits: {
                        where: {
                            is_deleted: false
                        },
                        select: ["name"],
                        populate: {
                            venue: {
                                select: ["name"]
                            }
                        }
                    },
                    ticket_type: true
                }
            });
            const groupedByDay = data.reduce((acc, item) => {
                const day = item.day;
                const existingDay = acc.find(entry => entry.day === day);
                if (existingDay) {
                    existingDay.benefits.push(...item.benefits);
                }
                else {
                    acc.push({ day: day, benefits: [...item.benefits] });
                }
                return acc;
            }, []);
            const result = groupedByDay.sort((a, b) => {
                const dayA = parseInt(a.day.match(/Day (\d+)/)[1], 10);
                const dayB = parseInt(b.day.match(/Day (\d+)/)[1], 10);
                return dayA - dayB;
            });
            return new dataResponse_dto_1.DataResponseDto(result, "Benefits fetched successfully");
        }
        catch (error) {
            console.log(error);
            return new dataResponse_dto_1.DataResponseDto(null, "Something went wrong in our end please contact system administrator");
        }
    },
    async createTicketBenefit(data) {
        try {
            // Check for unique constraint
            const exists = await this.checkUniqueConstraint({
                badgeCategory: data.badge_category,
                ticket_type: data.ticket_type,
                day: data.day,
            });
            if (!exists.isUnique) {
                return {
                    status: false,
                    message: "A ticket benefit with these details already exists",
                    data: [],
                };
            }
            const entry = await strapi.entityService.create("api::ticket-benefit.ticket-benefit", {
                data,
                populate: ["benefits", "ticket_type"],
            });
            return {
                status: true,
                message: "Ticket benefits created successfully",
                data: entry,
            };
        }
        catch (error) {
            console.error("Error in createTicketBenefit:", error);
            throw error;
        }
    },
    async checkUniqueConstraint(data, excludeId) {
        try {
            const { badge_category, ticket_type, day } = data;
            const filters = {
                badgeCategory: badge_category,
                ticket_type,
                day,
                deletedAt: null,
            };
            // If excludeId is provided, exclude that record from the check
            if (excludeId) {
                filters.id = {
                    $ne: excludeId,
                };
            }
            const existing = await strapi.entityService.findMany("api::ticket-benefit.ticket-benefit", {
                filters,
            });
            return {
                isUnique: existing.length === 0,
                existing: existing.length > 0 ? existing[0] : null,
            };
        }
        catch (error) {
            console.error("Error in checkUniqueConstraint:", error);
            throw error;
        }
    },
}));
