const puppeteer = require("puppeteer");
const fs = require("fs");
const path = require("path");
const invoiceTemplate = require("../templates/invoiceTemplate");
const { BlobServiceClient, StorageSharedKeyCredential } = require("@azure/storage-blob");
import moment from "moment";
const { toWords } = require("number-to-words");

module.exports = ({ strapi }) => ({
  async generateInvoice(orderData) {
    try {
      // Generate HTML from template
        const amountInWords = toWords(orderData.totalAmount).replace(/^\w/, c => c.toUpperCase());
         orderData.amountInWords = `${amountInWords} only`;

      const html = invoiceTemplate(orderData);

      // Launch puppeteer
      const browser = await puppeteer.launch({ headless: true });
      const page = await browser.newPage();
      await page.setContent(html, { waitUntil: "networkidle0" });

      // Generate PDF as buffer (not saved locally)
      const pdfBuffer = await page.pdf({
        format: "A4",
        printBackground: true,
      });

      await browser.close();

      // Prepare Azure upload configs
      const account = process.env.AZURE_STORAGE_ACCOUNT;
      const accountKey = process.env.AZURE_STORAGE_ACCOUNT_KEY;
      const containerName = process.env.AZURE_STORAGE_CONTAINER;
      const azureUrl = process.env.AZURE_STORAGE_URL;

      const sharedKeyCredential = new StorageSharedKeyCredential(account, accountKey);
      const blobServiceClient = new BlobServiceClient(`${azureUrl}`, sharedKeyCredential);
      const containerClient = blobServiceClient.getContainerClient(containerName);

      // File naming
      const fileName = `invoice-${orderData.orderId}.pdf`;
      const blobName = `invoices/pdf/${moment().format("YYYY-MM-DDHHmmss")}-${fileName}`.replace(/\s+/g, "_");

      const blockBlobClient = containerClient.getBlockBlobClient(blobName);

      // Upload PDF buffer
      await blockBlobClient.uploadData(pdfBuffer, {
        blobHTTPHeaders: {
          blobContentType: "application/pdf",
        },
      });

      // Final file URL from Azure
      const fileUrl = `${azureUrl}/${containerName}/${blobName}`;

      return fileUrl; // ye return karega Azure ka direct URL
    } catch (error) {
      console.error("Error generating or uploading invoice:", error.message);
      throw new Error("Failed to generate/upload invoice: " + error.message);
    }
  },
});
