import { factories } from "@strapi/strapi";
import { DataResponseDto } from "../../../shared/dto/dataResponse.dto";
import { SPEAKER_VALUES } from "../../../shared/helpers/constants";
const bcrypt = require("bcryptjs");
const { v4: uuidv4 } = require("uuid");
const { sanitize } = require("@strapi/utils");

async function generateOrderNum(type: string) {
  try {
    const uuid = uuidv4();
    const base36 = parseInt(uuid.replace(/-/g, ""), 16).toString(36);
    let firstLetter = type === "others" ? "O" : "S";
    const orderNumber = firstLetter + base36.slice(0, 7).toUpperCase();
    const isOrderNumberExists = await strapi.db
      .query("api::order.order")
      .findOne({
        where: { orderId: orderNumber, is_deleted: false },
      });
    if (isOrderNumberExists) return await generateOrderNum(type);
    return orderNumber;
  } catch (error) {
    console.log("error in generating order number", error);
  }
}

export default factories.createCoreService(
  "api::speaker.speaker",
  ({ strapi }) => ({
    async findExportData() {
      try {
        const data = await strapi.db.query("api::speaker.speaker").findMany({
          where: { is_deleted: false },
          orderBy: { createdAt: "desc" },
        });

        const model = strapi.getModel("api::speaker.speaker");
        const sanitizedData = await Promise.all(
          data.map((item) => sanitize.contentAPI.output(item, model))
        );


        return new DataResponseDto(sanitizedData, "Speaker data fetched successfully");
      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null, "Something went wrong in our end please contact system administrator"
        )
      }
    },
    async findAllSpeakers(params: any) {
      try {
        const { page, pageSize, search, filters } = params;
        let limit = Number(pageSize) || 10;
        let pageNumber = Number(page) || 1;

        const offset = (Number(pageNumber) - 1) * Number(limit);

        const where = {
          is_deleted: false,
          ...(filters && Object.keys(filters).length > 0 && { ...filters }),
          ...(search &&
            search.trim().length && {
            $or: [
              { firstName: { $containsi: search } },
              { lastName: { $containsi: search } },
              //  expect only two words
              {
                $and: [
                  { firstName: { $containsi: search.split(' ')[0] } },
                  { lastName: { $containsi: search.split(' ')[1] } },
                ],
              },
              { email: { $containsi: search } },
              { phone: { $containsi: search } },
              { companyName: { $containsi: search } },
              { nationality: { $containsi: search } },
            ],
          }),
        };
        const data = await strapi.db.query("api::speaker.speaker").findMany({
          where,
          offset,
          limit: limit,
          orderBy: { id: "desc" },
        });

        const model = strapi.getModel("api::speaker.speaker");
        const sanitizedData = await Promise.all(
          data.map((item) => sanitize.contentAPI.output(item, model))
        );

        if (sanitizedData) {
          const totalSpeakers = await strapi
            .query("api::speaker.speaker")
            .count({ where });
          const pageCount = Math.ceil(totalSpeakers / limit);
          return new DataResponseDto(
            {
              data: sanitizedData,
              pagination: {
                page: Number(pageNumber),
                pageSize: Number(limit),
                pageCount,
                total: totalSpeakers,
              },
            },
            "Speaker data fetched successfully"
          );
        } else {
          return new DataResponseDto(null);
        }
      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
    async picker(query: string) {
      try {
        const data = await strapi.db.query("api::speaker.speaker").findMany({
          where: query
            ? {
              is_deleted: false,
              $or: [
                { lastName: { $contains: query } },
                { firstName: { $contains: query } },
              ],
            }
            : { is_deleted: false },
          select: [
            "id",
            "firstName",
            "lastName",
            "companyName",
            "nationality",
            "residentCountry",
          ],
        });
        if (data) {
          return new DataResponseDto(data, "Speaker data fetched successfully");
        } else {
          return new DataResponseDto(null);
        }
      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
    async speakersGroupedByDate() {
      try {
        const sevenDaysAgo = new Date();
        sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);

        const data = await strapi.db.query("api::speaker.speaker").findMany({
          where: {
            is_deleted: false,
            created_at: {
              $gte: sevenDaysAgo,
              $lte: new Date(),
            },
          },
        });

        const model = strapi.getModel("api::speaker.speaker");
        const sanitizedData = await Promise.all(
          data.map((item) => sanitize.contentAPI.output(item, model))
        );
        if (sanitizedData) {
          return new DataResponseDto(sanitizedData, "Speaker data fetched successfully");
        } else {
          return new DataResponseDto(null);
        }
      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
    async latestSpeakers() {
      try {
        const sevenDaysAgo = new Date();
        sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);

        const data = await strapi.db.query("api::speaker.speaker").findMany({
          where: {
            is_deleted: false,
            created_at: {
              $gte: sevenDaysAgo,
              $lte: new Date(),
            },
          },
        });

        const model = strapi.getModel("api::speaker.speaker");
        const sanitizedData = await Promise.all(
          data.map((item) => sanitize.contentAPI.output(item, model))
        );

        if (sanitizedData) {
          return new DataResponseDto(sanitizedData, "Speaker data fetched successfully");
        } else {
          return new DataResponseDto(null);
        }
      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
    async softDelete(id: number) {
      try {
        const data = await strapi.db.query("api::speaker.speaker").update({
          where: { id, is_deleted: false },
          data: {
            is_deleted: true,
          },
        });

        const model = strapi.getModel("api::speaker.speaker");
        const sanitizedData = await sanitize.contentAPI.output(data, model);
        if (sanitizedData) {
          return new DataResponseDto(sanitizedData, "Speaker deleted successfully");
        } else {
          return new DataResponseDto(null);
        }
      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
    async speakerLogin(body: any) {
      try {
        const { speakerId, password } = body;
        if (!speakerId || !password) {
          return new DataResponseDto(null, "Please provide valid credentials");
        }
        const speaker = await strapi.query("api::speaker.speaker").findOne({
          where: { id: speakerId, is_deleted: false },
        });

        const model = strapi.getModel("api::speaker.speaker");
        const sanitizedData = await sanitize.contentAPI.output(speaker, model);


        if (!sanitizedData) {
          return {
            data: null,
            status: false,
            message: "Speaker not found.",
          };
        }

        const isPasswordValid = await bcrypt.compare(
          password,
          sanitizedData?.password
        );

        if (!isPasswordValid) {
          return {
            data: null,
            status: false,
            message: "Invalid credentials.",
          };
        }

        return new DataResponseDto(sanitizedData, "Speaker logged in successfully");
      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
    async createSpeaker(body: any) {
      try {
        let orderNumber = await generateOrderNum('other');

        const order = await strapi.entityService.create("api::order.order", {
          data: {
            orderId: orderNumber,
            type: 'others',
            paymentId: null,
            eventDate: new Date(),
            firstName: body.firstName,
            lastName: body.lastName,
            email: body.email,
            company: body.companyName,
            // sector: body.sector, // department  
            designation: body.designation,
            nationality: body.nationality,
            residanceCountry: body.residentCountry,
            // countryCode: body.countryCode,
            mobile: body.mobile,
            photo: body.photoUrl,
            ticketCount: SPEAKER_VALUES.TOTAL_TICKET_COUNT,
            discountAmount: 0,
            totalAmount: 0,
            paidAmount: 0,
            is_discounted: false,
            publishedAt: new Date(),
            emirate: body?.emirate,
            status: "Free" as "Free",
            userType: "speaker" as "speaker",
            creator: body?.userId,
          },
        });

        let password = await bcrypt.hash(SPEAKER_VALUES.PASSWORD.trim(), 10);

        const speaker = await strapi.db.query("api::speaker.speaker").create({
          data: {
            designation: body?.designation,
            department: body?.department,
            photoUrl: body?.photoUrl,
            firstName: body?.firstName,
            lastName: body?.lastName,
            email: body?.email,
            companyName: body?.companyName,
            bio: body?.bio,
            nationality: body?.nationality,
            residentCountry: body?.residentCountry,
            graduationYear: body?.graduationYear,
            uniName: body?.uniName,
            passportUrl: body?.passportUrl,
            emiratesIdFront: body?.emiratesIdFront,
            phone: body?.phone,
            qualification: body?.qualification,
            emiratesIdBack: body?.emiratesIdBack,
            published: false,
            social: body?.social,
            socialDescription: body?.socialDescription,
            countryCode: body?.countryCode,
            emirate: body?.emirate,
            priority: body?.priority,
            creator: body?.userId,
            modifiedBy: body?.userId,
            mobile_app_photo_url: body?.mobile_app_photo_url,

            marital_status: body?.marital_status,
            religion: body?.religion,
            faith: body?.faith,
            indexing:body?.indexing || 200,

            password: password,
            single_day: SPEAKER_VALUES.SINGLE_DAY_COUNT,
            executive: SPEAKER_VALUES.EXECUTIVE_COUNT,
            delegate: SPEAKER_VALUES.DELEGATE_COUNT,
            // padock_club: SPEAKER_VALUES.PADOCK_CLUB_COUNT,
            // general_admission: SPEAKER_VALUES.GENERAL_ADMISSION_COUNT,
            resolve: SPEAKER_VALUES.RESOLVE_COUNT,
            asset_abu_dhabi: SPEAKER_VALUES.ASSET_ABU_DHABI_COUNT,
            fintech_abu_dhabi: SPEAKER_VALUES.FINTECH_ABU_DHABI_COUNT,
            adsff: SPEAKER_VALUES.ADSFF_COUNT,

            remaining_executive: SPEAKER_VALUES.EXECUTIVE_COUNT,
            remaining_delegate: SPEAKER_VALUES.DELEGATE_COUNT,
            // remaining_padock_club: SPEAKER_VALUES.PADOCK_CLUB_COUNT,
            // remaining_general_admission: SPEAKER_VALUES.GENERAL_ADMISSION_COUNT,
            remaining_resolve: SPEAKER_VALUES.RESOLVE_COUNT,
            remaining_asset_abu_dhabi: SPEAKER_VALUES.ASSET_ABU_DHABI_COUNT,
            remaining_fintech_abu_dhabi: SPEAKER_VALUES.FINTECH_ABU_DHABI_COUNT,
            remaining_adsff: SPEAKER_VALUES.ADSFF_COUNT,
            remaining_single_day: SPEAKER_VALUES.SINGLE_DAY_COUNT,
            order: order.id,
            publishedAt: new Date(),
            visaCopy: body?.visaCopy,
          }
        })

        await strapi.entityService.update(
          "api::order.order",
          order.id,
          {
            data: {
              speaker: speaker.id
            },
          }
        );
        return new DataResponseDto("Speaker created successfully");

      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
    async updateSpeakerTicketDetails(id: number, body: any) {
      try {
        const existingSpeaker = await strapi.db
          .query("api::speaker.speaker")
          .findOne({
            where: { id },
          });
        if (!existingSpeaker) {
          return new DataResponseDto(null, "Speaker not found");
        }
        let order;
        if (body?.orderId) {
          order = await strapi.db
            .query("api::order.order")
            .findOne({
              where: { id: body?.orderId },
            });
          if (!order) {
            return new DataResponseDto(null, "Order not found");
          }
        } else {
          let orderNumber = await generateOrderNum('other');
          order = await strapi.entityService.create("api::order.order", {
            data: {
              orderId: orderNumber,
              type: 'others',
              paymentId: null,
              eventDate: new Date(),
              firstName: existingSpeaker.firstName,
              lastName: existingSpeaker.lastName,
              email: existingSpeaker.email,
              company: existingSpeaker.companyName,
              // sector: body.sector, // department  
              designation: existingSpeaker.designation,
              nationality: existingSpeaker.nationality,
              residanceCountry: existingSpeaker.residentCountry,
              // countryCode: body.countryCode,
              mobile: existingSpeaker.mobile,
              photo: existingSpeaker.photoUrl,

              ticketCount: body?.ticketCount,
              discountAmount: 0,
              totalAmount: 0,
              paidAmount: 0,
              is_discounted: false,
              publishedAt: new Date(),
              emirate: existingSpeaker?.emirate,
              status: "Free" as "Free",
              userType: "speaker" as "speaker",
              speaker: existingSpeaker.id
            },
          });
        }

        //  remaining count
        let remaining_executive_count: number =
          Number(existingSpeaker?.remaining_executive) || 0; // executive

        let remaining_delegate_count: number =
          Number(existingSpeaker?.remaining_delegate) || 0; // delegate

        // let remaining_padock_club_count: number =
        //   Number(existingSpeaker?.remaining_padock_club) || 0; // padock club

        // let remaining_general_admission_count: number =
        //   Number(existingSpeaker?.remaining_general_admission) || 0;  // general admission

        let remaining_single_day_delegate_count: number =
          Number(existingSpeaker?.remaining_single_day) || 0; // single day

        let remaining_resolve_count: number =
          Number(existingSpeaker?.remaining_resolve) || 0; // resolve

        let remaining_asset_abu_dhabi_count: number =
          Number(existingSpeaker?.remaining_asset_abu_dhabi) || 0;  // asset abu dhabi

        let remaining_fintech_abu_dhabi_count: number =
          Number(existingSpeaker?.remaining_fintech_abu_dhabi) || 0; // remaining_fintech_abu_dhabi

        let remaining_adsff_count: number =
          Number(existingSpeaker?.remaining_adsff) || 0; // remaining_adsff

        // exective
        if (body.executive === 0 || body.executive) {
          let executive_count = body.executive === 0 ? 0 : body.executive;
          const executive_remaining =
            Number(existingSpeaker?.executive) -
            Number(existingSpeaker?.remaining_executive);

          if (Number(executive_remaining) > Number(executive_count)) {
            return new DataResponseDto(
              null,
              "executive cannot be less than redeemed tickets count"
            );
          }
          remaining_executive_count = Number(executive_count) - Number(executive_remaining);
        }


        // full week
        if (body.delegate === 0 || body.delegate) {
          let delegate_count = body.delegate === 0 ? 0 : body.delegate;
          let delegate_remaining =
            Number(existingSpeaker?.delegate) -
            Number(existingSpeaker?.remaining_delegate);

          if (Number(delegate_remaining) > Number(delegate_count)) {
            return new DataResponseDto(
              null,
              "total full week count cannot be less than redeemed tickets count"
            );
          }
          remaining_delegate_count =
            Number(delegate_count) - Number(delegate_remaining);
        }


        // padock_club
        // if (body.padock_club === 0 || body.padock_club) {
        //   let padock_club_count = body.padock_club === 0 ? 0 : body.padock_club;
        //   let padock_club_remaining =
        //     Number(existingSpeaker?.padock_club) -
        //     Number(existingSpeaker?.remaining_padock_club);

        //   if (Number(padock_club_remaining) > Number(padock_club_count)) {
        //     return new DataResponseDto(
        //       null,
        //       "total padock club count cannot be less than redeemed tickets count"
        //     );
        //   }
        // remaining_padock_club_count =
        //   Number(padock_club_count) - Number(padock_club_remaining);
        // }


        // general_admission
        // if (body.general_admission === 0 || body.general_admission) {
        //   let general_admission_counnt = body.general_admission === 0 ? 0 : body.general_admission;
        //   let general_admission_remaining =
        //     Number(existingSpeaker?.general_admission) -
        //     Number(existingSpeaker?.remaining_general_admission);

        //   if (
        //     Number(general_admission_remaining) > Number(general_admission_counnt)
        //   ) {
        //     return new DataResponseDto(
        //       null,
        //       "total general admission count cannot be less than redeemed tickets count"
        //     );
        //   }
        //   remaining_general_admission_count =
        //     Number(general_admission_counnt) -
        //     Number(general_admission_remaining);

        // }

        // single_day_delegate
        if (body.single_day === 0 || body.single_day) {
          let single_day_delegate_count = body.single_day === 0 ? 0 : body.single_day;
          let single_day_delegate_remaining =
            Number(existingSpeaker?.single_day) -
            Number(existingSpeaker?.remaining_single_day);

          if (
            Number(single_day_delegate_remaining) >
            Number(single_day_delegate_count)
          ) {
            return new DataResponseDto(
              null,
              "total single day delegate count cannot be less than redeemed tickets count"
            );
          }
          remaining_single_day_delegate_count =
            Number(single_day_delegate_count) -
            Number(single_day_delegate_remaining);

        }


        // resolve done
        if (body.resolve === 0 || body.resolve) {
          let resolveCount = body.resolve === 0 ? 0 : body.resolve;
          let resolve_remaining =
            Number(existingSpeaker?.resolve) -
            Number(existingSpeaker?.remaining_resolve);
          if (Number(resolve_remaining) > Number(resolveCount)) {
            return new DataResponseDto(
              null,
              "total resolve count cannot be less than redeemed tickets count"
            );
          }
          remaining_resolve_count =
            Number(resolveCount) - Number(resolve_remaining);
        }


        // asset_abu_dhabi
        if (body.asset_abu_dhabi === 0 || body.asset_abu_dhabi) {
          let asset_abu_dhabiCount = body.asset_abu_dhabi === 0 ? 0 : body.asset_abu_dhabi;
          let asset_abu_dhabi_remaining =
            Number(existingSpeaker?.asset_abu_dhabi) -
            Number(existingSpeaker?.remaining_asset_abu_dhabi);
          if (
            Number(asset_abu_dhabi_remaining) > Number(asset_abu_dhabiCount)
          ) {
            return new DataResponseDto(
              null,
              "total asset_abu_dhabi count cannot be less than redeemed tickets count"
            );
          }
          remaining_asset_abu_dhabi_count =
            Number(asset_abu_dhabiCount) - Number(asset_abu_dhabi_remaining);

        }

        // fintech_abu_dhabi
        if (body.fintech_abu_dhabi === 0 || body.fintech_abu_dhabi) {
          let fintech_abu_dhabiCount = body.fintech_abu_dhabi === 0 ? 0 : body.fintech_abu_dhabi;
          let fintech_abu_dhabi_remaining =
            Number(existingSpeaker.fintech_abu_dhabi) -
            Number(existingSpeaker.remaining_fintech_abu_dhabi);
          if (
            Number(fintech_abu_dhabi_remaining) > Number(fintech_abu_dhabiCount)
          ) {
            return new DataResponseDto(
              null,
              "total fintech_abu_dhabi count cannot be less than redeemed tickets count"
            );
          }
          remaining_fintech_abu_dhabi_count =
            Number(fintech_abu_dhabiCount) -
            Number(fintech_abu_dhabi_remaining);
        }

        // adsff

        if (body.adsff === 0 || body.adsff) {
          let adsff_Count = body.adsff === 0 ? 0 : body.adsff;
          let adsff_remaining =
            Number(existingSpeaker?.adsff) -
            Number(existingSpeaker?.remaining_adsff);
          if (Number(adsff_remaining) > Number(adsff_Count)) {
            return new DataResponseDto(
              null,
              "total adsff count cannot be less than redeemed tickets count"
            );
          }
          remaining_adsff_count =
            Number(adsff_Count) - Number(adsff_remaining);
        }

        // hash new password
        let password = existingSpeaker?.password;
        if (body.password) {
          password = await bcrypt.hash(body.password.toString().trim(), 10);
        }

        // let padock_club = existingSpeaker?.padock_club;
        // if (body?.padock_club === 0 || body?.padock_club) {
        //   padock_club = body?.padock_club === 0 ? 0 : Number(body?.padock_club)
        // }


        // update speaker
        const speaker = await strapi.db.query("api::speaker.speaker").update({
          where: { id },
          data: {
            password: password,
            single_day: body?.single_day === 0 ? 0 : Number(body?.single_day),

            executive: Number(body.executive) === 0 ? 0 : Number(body?.executive),

            delegate: Number(body.delegate) === 0 ? 0 : Number(body?.delegate),

            // padock_club: Number(padock_club),

            // general_admission:
            //   Number(body.general_admission) === 0 ? 0 : Number(body?.general_admission),

            resolve: Number(body.resolve) === 0 ? 0 : Number(body?.resolve),

            asset_abu_dhabi:
              Number(body?.asset_abu_dhabi) === 0 ? 0 : Number(body?.asset_abu_dhabi),

            fintech_abu_dhabi:
              Number(body.fintech_abu_dhabi) === 0 ? 0 : Number(body?.fintech_abu_dhabi),

            adsff: Number(body.adsff) === 0 ? 0 : Number(body?.adsff),

            remaining_executive: Number(remaining_executive_count),
            remaining_delegate: Number(remaining_delegate_count),
            // remaining_padock_club: Number(remaining_padock_club_count),
            // remaining_general_admission: Number(
            //   remaining_general_admission_count
            // ),
            remaining_resolve: Number(remaining_resolve_count),
            remaining_asset_abu_dhabi: Number(
              remaining_asset_abu_dhabi_count
            ),
            remaining_fintech_abu_dhabi: Number(
              remaining_fintech_abu_dhabi_count
            ),
            remaining_adsff: Number(remaining_adsff_count),
            remaining_single_day: Number(
              remaining_single_day_delegate_count
            ),
            order: order.id
          }
        })
        
        const model = strapi.getModel("api::speaker.speaker");
        const sanitizedData = await sanitize.contentAPI.output(speaker, model);

        return new DataResponseDto(sanitizedData, "speaker Updated successfully");
      } catch (error) {
        console.log(error)
        return new DataResponseDto(null, "Something went wrong in our end please contact system administrator");
      }
    },
    async bulkCreateSpeakers(bodies: any[]) {
      try {
        if (!bodies.length) return new DataResponseDto(null, "Please provide valid data");

        const createdSpeakers = [];
        for (const body of bodies) {
          let orderNumber = await generateOrderNum('other');
          const order = await strapi.entityService.create("api::order.order", {
            data: {
              orderId: orderNumber,
              type: 'others',
              paymentId: null,
              eventDate: new Date(),
              firstName: body.firstName,
              lastName: body.lastName,
              email: body.email,
              company: body.companyName,
              // sector: body.sector, // department  
              designation: body.designation,
              nationality: body.nationality,
              residanceCountry: body.residentCountry,
              // countryCode: body.countryCode,
              mobile: body.mobile,
              photo: body.photoUrl,
              ticketCount: SPEAKER_VALUES.TOTAL_TICKET_COUNT,
              discountAmount: 0,
              totalAmount: 0,
              paidAmount: 0,
              is_discounted: false,
              publishedAt: new Date(),
              emirate: body?.emirate,
              status: "Free" as "Free",
              userType: "speaker" as "speaker",
              creator: body?.userId,
            },
          });

          let password = await bcrypt.hash(SPEAKER_VALUES.PASSWORD.trim(), 10);

          const speaker = await strapi.db.query("api::speaker.speaker").create({
            data: {
              designation: body?.designation,
              department: body?.department,
              photoUrl: body?.photoUrl,
              firstName: body?.firstName,
              lastName: body?.lastName,
              email: body?.email,
              companyName: body?.companyName,
              bio: body?.bio,
              nationality: body?.nationality,
              residentCountry: body?.residentCountry,
              graduationYear: body?.graduationYear,
              uniName: body?.uniName,
              passportUrl: body?.passportUrl,
              emiratesIdFront: body?.emiratesIdFront,
              phone: body?.phone,
              qualification: body?.qualification,
              emiratesIdBack: body?.emiratesIdBack,
              published: body?.published,
              social: body?.social,
              socialDescription: body?.socialDescription,
              countryCode: body?.countryCode,
              emirate: body?.emirate,
              priority: body?.priority || 100,
              creator: body?.userId,
              modifiedBy: body?.userId,
              mobile_app_photo_url: body?.mobile_app_photo_url,

              marital_status: body?.marital_status,
              religion: body?.religion,
              faith: body?.faith,

              password: password,
              single_day: SPEAKER_VALUES.SINGLE_DAY_COUNT,
              executive: SPEAKER_VALUES.EXECUTIVE_COUNT,
              delegate: SPEAKER_VALUES.DELEGATE_COUNT,
              // padock_club: SPEAKER_VALUES.PADOCK_CLUB_COUNT,
              // general_admission: SPEAKER_VALUES.GENERAL_ADMISSION_COUNT,
              resolve: SPEAKER_VALUES.RESOLVE_COUNT,
              asset_abu_dhabi: SPEAKER_VALUES.ASSET_ABU_DHABI_COUNT,
              fintech_abu_dhabi: SPEAKER_VALUES.FINTECH_ABU_DHABI_COUNT,
              adsff: SPEAKER_VALUES.ADSFF_COUNT,

              remaining_executive: SPEAKER_VALUES.EXECUTIVE_COUNT,
              remaining_delegate: SPEAKER_VALUES.DELEGATE_COUNT,
              // remaining_padock_club: SPEAKER_VALUES.PADOCK_CLUB_COUNT,
              // remaining_general_admission: SPEAKER_VALUES.GENERAL_ADMISSION_COUNT,
              remaining_resolve: SPEAKER_VALUES.RESOLVE_COUNT,
              remaining_asset_abu_dhabi: SPEAKER_VALUES.ASSET_ABU_DHABI_COUNT,
              remaining_fintech_abu_dhabi: SPEAKER_VALUES.FINTECH_ABU_DHABI_COUNT,
              remaining_adsff: SPEAKER_VALUES.ADSFF_COUNT,
              remaining_single_day: SPEAKER_VALUES.SINGLE_DAY_COUNT,
              order: order.id,
              publishedAt: null,
            }
          })

          await strapi.entityService.update(
            "api::order.order",
            order.id,
            {
              data: {
                speaker: speaker.id
              },
            }
          );
          createdSpeakers.push(speaker);
        }
        return new DataResponseDto("Speakers created successfully");

      } catch (error) {
        console.log(error);
        return new DataResponseDto(
          null,
          "Something went wrong in our end please contact system administrator"
        );
      }
    },
  })
);
